<?php

namespace App\Http\Controllers;

use App\Models\BudgetEntry;
use App\Models\BudgetLogBook;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class BudgetEntryController extends Controller
{

    // Status mapping - Updated for consistency
    private $statusMap = [
        'Draft' => 'D',
        'Revised' => 'R',
        'Approve' => 'P',
        'Freeze' => 'F',
        'Active' => 'A',
        'Inactive' => 'I',
        'Re-Revised' => 'RR',
        'Reject' => 'RJ',
        'Justification' => 'J'
    ];

    private $reverseStatusMap = [
        'D' => 'Draft',
        'R' => 'Revised',
        'P' => 'Approve',
        'F' => 'Freeze',
        'A' => 'Active',
        'I' => 'Inactive',
        'RR' => 'Re-Revised',  // Consistent naming
        'RJ' => 'Reject',
        'J' => 'Justification'
    ];

    private function convertToStatusCode($status)
    {
        // Handle both Re_Revised and Re-Revised for backward compatibility
        if ($status === 'Re_Revised' || $status === 'Re-Revised') {
            return 'RR';
        }

        return $this->statusMap[$status] ?? $status;
    }

    private function convertToStatusText($statusCode)
    {
        return $this->reverseStatusMap[$statusCode] ?? $statusCode;
    }


    public function index(Request $request)
{
    try {
        $query = BudgetEntry::query();

        // Apply filters
        if ($request->has('fin_year')) {
            $query->where('Fin_Year', $request->fin_year);
        }

        if ($request->has('dept_id')) {
            $query->where('Dept_Id', $request->dept_id);
        }

        if ($request->has('ledg_id')) {
            $query->where('Ledg_ID', $request->ledg_id);
        }

        if ($request->has('status')) {
            $status = $this->convertToStatusCode($request->status);
            $query->where('Status', $status);
        }

        if ($request->has('budg_id')) {
            $query->where('budget_entries.Budg_Id', $request->budg_id);
        }

        // Filter by ref_id if provided
        if ($request->has('ref_id')) {
            $refId = $request->ref_id;
            if ($refId == '0' || $refId == 0) {
                $query->where('Ref_Id', 0);
            } else {
                $query->where('Ref_Id', $refId);
            }
        }

        // Filter by is_latest if provided
        if ($request->has('is_latest')) {
            $isLatest = filter_var($request->is_latest, FILTER_VALIDATE_BOOLEAN);
            $query->where('Is_Latest', $isLatest);
        }

        // Show only latest versions if requested (backward compatibility)
        if ($request->has('show_only_latest') && $request->show_only_latest == true) {
            $query->where('Is_Latest', true);
        }

        // Show only root entries (first versions) if requested
        if ($request->has('show_only_root') && $request->show_only_root == true) {
            $query->where('Ref_Id', 0);
        }

        // Load relationships including parent entry with full details
        $with = ['financialYear', 'department', 'ledger' => function ($q) {
            $q->with('ledgerGroup');
        }];

        // Add parent entry if ref_id > 0
        $query->with(array_merge($with, [
            'parentEntry' => function ($q) use ($with) {
                $q->with($with);
            }
        ]));

        // Sorting
        $sortField = $request->get('sort_by', 'Budg_Id');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortField, $sortOrder);

        // Pagination
        $perPage = $request->get('per_page', 15);
        $budgetEntries = $query->paginate($perPage);

        // Format response data
        $formattedData = $budgetEntries->map(function ($item) {
            return $this->formatBudgetEntryData($item);
        });

        // Calculate totals
        $totalAmount = $budgetEntries->sum('Amount');
        $totalSugAmount = $budgetEntries->sum('Sug_Amount');

        return response()->json([
            'success' => true,
            'message' => 'Budget entries retrieved successfully',
            'data' => $formattedData,
            'summary' => [
                'total_amount' => (float) $totalAmount,
                'formatted_total_amount' => '₹ ' . number_format($totalAmount, 2),
                'total_sug_amount' => (float) $totalSugAmount,
                'formatted_total_sug_amount' => '₹ ' . number_format($totalSugAmount, 2),
                'total_records' => $budgetEntries->total(),
                'latest_records_count' => $budgetEntries->where('Is_Latest', true)->count(),
                'non_latest_records_count' => $budgetEntries->where('Is_Latest', false)->count(),
            ],
            'pagination' => [
                'current_page' => $budgetEntries->currentPage(),
                'per_page' => $budgetEntries->perPage(),
                'total' => $budgetEntries->total(),
                'last_page' => $budgetEntries->lastPage(),
            ]
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to retrieve budget entries',
            'error' => $e->getMessage()
        ], 500);
    }
}



/**
 * Get count of budget entries grouped by status (all records by default)
 */
public function getStatusCounts(Request $request)
{
    try {
        $query = BudgetEntry::query();

        // Apply filters
        if ($request->has('fin_year')) {
            $query->where('Fin_Year', $request->fin_year);
        }

        if ($request->has('dept_id')) {
            $query->where('Dept_Id', $request->dept_id);
        }

        if ($request->has('ledg_id')) {
            $query->where('Ledg_ID', $request->ledg_id);
        }

        if ($request->has('budg_id')) {
            $query->where('budget_entries.Budg_Id', $request->budg_id);
        }

        // Filter by is_latest if provided
        if ($request->has('is_latest')) {
            $isLatest = filter_var($request->is_latest, FILTER_VALIDATE_BOOLEAN);
            $query->where('Is_Latest', $isLatest);
        }

        // Count only latest versions if requested (backward compatibility)
        if ($request->has('count_only_latest') && $request->count_only_latest == true) {
            $query->where('Is_Latest', true);
        }

        // Count only root entries if requested
        if ($request->has('count_only_root') && $request->count_only_root == true) {
            $query->where('Ref_Id', 0);
        }

        // Get all status counts
        $statusCounts = $query->select('Status', DB::raw('COUNT(*) as count'))
            ->groupBy('Status')
            ->get()
            ->mapWithKeys(function ($item) {
                return [$item->Status => $item->count];
            })
            ->toArray();

        // Get total count
        $totalCount = array_sum($statusCounts);

        // Also get counts for latest and non-latest separately
        $latestCount = $query->clone()->where('Is_Latest', true)->count();
        $nonLatestCount = $query->clone()->where('Is_Latest', false)->count();

        // Format the response with all statuses (including zero counts)
        $allStatuses = [
            'D' => 'Draft',
            'R' => 'Revised',
            'P' => 'Approve',
            'F' => 'Freeze',
            'A' => 'Active',
            'I' => 'Inactive',
            'RR' => 'Re-Revised',
            'RJ' => 'Reject',
            'J' => 'Justification'
        ];

        $formattedCounts = [];
        $totalAmount = 0;
        $totalSugAmount = 0;
        $latestTotalAmount = 0;
        $latestTotalSugAmount = 0;

        foreach ($allStatuses as $code => $text) {
            $count = $statusCounts[$code] ?? 0;

            // Calculate total amount and suggested amount for each status
            if ($count > 0) {
                $statusQuery = BudgetEntry::query();

                // Reapply filters for this status
                if ($request->has('fin_year')) {
                    $statusQuery->where('Fin_Year', $request->fin_year);
                }
                if ($request->has('dept_id')) {
                    $statusQuery->where('Dept_Id', $request->dept_id);
                }
                if ($request->has('ledg_id')) {
                    $statusQuery->where('Ledg_ID', $request->ledg_id);
                }
                if ($request->has('budg_id')) {
                    $statusQuery->where('budget_entries.Budg_Id', $request->budg_id);
                }

                // Apply same filters for latest/root only
                if ($request->has('count_only_latest') && $request->count_only_latest == true) {
                    $statusQuery->where('Is_Latest', true);
                }
                if ($request->has('count_only_root') && $request->count_only_root == true) {
                    $statusQuery->where('Ref_Id', 0);
                }

                // Apply is_latest filter if provided
                if ($request->has('is_latest')) {
                    $isLatest = filter_var($request->is_latest, FILTER_VALIDATE_BOOLEAN);
                    $statusQuery->where('Is_Latest', $isLatest);
                }

                $statusQuery->where('Status', $code);
                $statusEntries = $statusQuery->get();

                $statusAmount = $statusEntries->sum('Amount');
                $statusSugAmount = $statusEntries->sum('Sug_Amount');

                // Calculate latest amounts separately
                $latestStatusEntries = $statusQuery->clone()->where('Is_Latest', true)->get();
                $latestStatusAmount = $latestStatusEntries->sum('Amount');
                $latestStatusSugAmount = $latestStatusEntries->sum('Sug_Amount');

                $totalAmount += $statusAmount;
                $totalSugAmount += $statusSugAmount;
                $latestTotalAmount += $latestStatusAmount;
                $latestTotalSugAmount += $latestStatusSugAmount;

                $formattedCounts[] = [
                    'status_code' => $code,
                    'status_text' => $text,
                    'count' => $count,
                    'amount' => (float) $statusAmount,
                    'formatted_amount' => '₹ ' . number_format($statusAmount, 2),
                    'sug_amount' => (float) $statusSugAmount,
                    'formatted_sug_amount' => '₹ ' . number_format($statusSugAmount, 2),
                    'latest_count' => $latestStatusEntries->count(),
                    'latest_amount' => (float) $latestStatusAmount,
                    'formatted_latest_amount' => '₹ ' . number_format($latestStatusAmount, 2),
                    'latest_sug_amount' => (float) $latestStatusSugAmount,
                    'formatted_latest_sug_amount' => '₹ ' . number_format($latestStatusSugAmount, 2),
                    'percentage' => $totalCount > 0 ? round(($count / $totalCount) * 100, 2) : 0,
                    'is_workflow_status' => !in_array($code, ['A', 'I']), // All except Active and Inactive
                    'is_active' => $code === 'A',
                    'can_edit' => in_array($code, ['D', 'R', 'RR', 'J']), // Editable statuses
                ];
            } else {
                $formattedCounts[] = [
                    'status_code' => $code,
                    'status_text' => $text,
                    'count' => 0,
                    'amount' => 0.00,
                    'formatted_amount' => '₹ 0.00',
                    'sug_amount' => 0.00,
                    'formatted_sug_amount' => '₹ 0.00',
                    'latest_count' => 0,
                    'latest_amount' => 0.00,
                    'formatted_latest_amount' => '₹ 0.00',
                    'latest_sug_amount' => 0.00,
                    'formatted_latest_sug_amount' => '₹ 0.00',
                    'percentage' => 0.00,
                    'is_workflow_status' => !in_array($code, ['A', 'I']),
                    'is_active' => $code === 'A',
                    'can_edit' => in_array($code, ['D', 'R', 'RR', 'J']),
                ];
            }
        }

        // Calculate workflow vs final status counts
        $workflowStatuses = array_filter($formattedCounts, function ($item) {
            return $item['is_workflow_status'];
        });
        $finalStatuses = array_filter($formattedCounts, function ($item) {
            return !$item['is_workflow_status'];
        });

        $workflowCount = array_sum(array_column($workflowStatuses, 'count'));
        $finalCount = array_sum(array_column($finalStatuses, 'count'));

        $latestWorkflowCount = array_sum(array_column($workflowStatuses, 'latest_count'));
        $latestFinalCount = array_sum(array_column($finalStatuses, 'latest_count'));

        return response()->json([
            'success' => true,
            'message' => 'Status counts retrieved successfully',
            'data' => $formattedCounts,
            'summary' => [
                'total_count' => $totalCount,
                'total_amount' => (float) $totalAmount,
                'formatted_total_amount' => '₹ ' . number_format($totalAmount, 2),
                'total_sug_amount' => (float) $totalSugAmount,
                'formatted_total_sug_amount' => '₹ ' . number_format($totalSugAmount, 2),
                'latest_count' => $latestCount,
                'non_latest_count' => $nonLatestCount,
                'latest_total_amount' => (float) $latestTotalAmount,
                'formatted_latest_total_amount' => '₹ ' . number_format($latestTotalAmount, 2),
                'latest_total_sug_amount' => (float) $latestTotalSugAmount,
                'formatted_latest_total_sug_amount' => '₹ ' . number_format($latestTotalSugAmount, 2),
                'workflow_status_count' => $workflowCount,
                'final_status_count' => $finalCount,
                'latest_workflow_count' => $latestWorkflowCount,
                'latest_final_count' => $latestFinalCount,
                'workflow_percentage' => $totalCount > 0 ? round(($workflowCount / $totalCount) * 100, 2) : 0,
                'final_percentage' => $totalCount > 0 ? round(($finalCount / $totalCount) * 100, 2) : 0,
                'latest_workflow_percentage' => $latestCount > 0 ? round(($latestWorkflowCount / $latestCount) * 100, 2) : 0,
                'latest_final_percentage' => $latestCount > 0 ? round(($latestFinalCount / $latestCount) * 100, 2) : 0,
            ],
            'filters_applied' => [
                'fin_year' => $request->fin_year ?? null,
                'dept_id' => $request->dept_id ?? null,
                'ledg_id' => $request->ledg_id ?? null,
                'budg_id' => $request->budg_id ?? null,
                'is_latest' => $request->has('is_latest') ? filter_var($request->is_latest, FILTER_VALIDATE_BOOLEAN) : null,
                'count_only_latest' => $request->count_only_latest ?? false,
                'count_only_root' => $request->count_only_root ?? false,
            ]
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to retrieve status counts',
            'error' => $e->getMessage()
        ], 500);
    }
}

    /**
     * Store a newly created budget entry
     */
    public function store(Request $request)
    {
        DB::beginTransaction();

        try {
            $validator = Validator::make($request->all(), [
                'Fin_Year' => 'required|integer|exists:financial_year_master,financial_year_id',
                'Dept_Id' => 'required|integer|exists:main_departments,id',
                'Ledg_ID' => 'required|integer|exists:ledger_master,Lg_Id',
                'Amount' => 'required|numeric|min:0',
                'Sug_Amount' => 'nullable|numeric|min:0',
                'Beg_Doc' => 'nullable|array',
                'Beg_Doc.*' => 'nullable|string|max:500',
                'Status' => 'sometimes|string|in:D,R,P,F,A,I,RR,RJ,J,Draft,Revised,Approve,Freeze,Active,Inactive,Re-Revised,Reject,Justification',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Convert status
            $status = $this->convertToStatusCode($request->Status ?? 'Draft');

            // Check if budget entry already exists (check only root entries)
            $existingEntry = BudgetEntry::where('Ref_Id', 0)
                ->where('Fin_Year', $request->Fin_Year)
                ->where('Dept_Id', $request->Dept_Id)
                ->where('Ledg_ID', $request->Ledg_ID)
                ->first();

            if ($existingEntry) {
                return response()->json([
                    'success' => false,
                    'message' => 'Budget entry already exists for this financial year, department and ledger combination'
                ], 409);
            }

            // Create new budget entry
            $budgetEntry = BudgetEntry::create([
                'Fin_Year' => $request->Fin_Year,
                'Dept_Id' => $request->Dept_Id,
                'Ledg_ID' => $request->Ledg_ID,
                'Amount' => $request->Amount,
                'Sug_Amount' => $request->Sug_Amount,
                'Beg_Doc' => $request->Beg_Doc,
                'Status' => $status,
                'Ref_Id' => 0, // First version
            ]);

            DB::commit();

            $budgetEntry->load(['financialYear', 'department', 'ledger', 'logEntries.user']);

            return response()->json([
                'success' => true,
                'message' => 'Budget entry created successfully',
                'data' => $this->formatBudgetEntryData($budgetEntry)
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to create budget entry',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified budget entry
     */
     public function show($id)
    {
        try {
            $budgetEntry = BudgetEntry::with([
                'financialYear',
                'department',
                'ledger' => function ($query) {
                    $query->with('ledgerGroup');
                },
                'logEntries.user',
                'parentEntry' => function ($query) {
                    $query->with(['financialYear', 'department', 'ledger' => function ($q) {
                        $q->with('ledgerGroup');
                    }]);
                },
                'childEntries' => function ($query) {
                    $query->with(['financialYear', 'department', 'ledger' => function ($q) {
                        $q->with('ledgerGroup');
                    }]);
                }
            ])
                ->find($id);

            if (!$budgetEntry) {
                return response()->json([
                    'success' => false,
                    'message' => 'Budget entry not found'
                ], 404);
            }

            // Get version history
            $versionHistory = $budgetEntry->getVersionHistory();

            return response()->json([
                'success' => true,
                'message' => 'Budget entry retrieved successfully',
                'data' => $this->formatBudgetEntryData($budgetEntry),
                'version_history' => $versionHistory->map(function ($version) {
                    return $this->formatBudgetEntryData($version);
                }),
                'version_info' => [
                    'version_number' => $budgetEntry->version_number,
                    'is_latest' => $budgetEntry->is_latest,
                    'root_id' => $budgetEntry->getRootEntryAttribute()->Budg_Id,
                    'total_versions' => $versionHistory->count(),
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve budget entry',
                'error' => $e->getMessage()
            ], 500);
        }
    }


 /**
 * Update budget entry (creates a new version) with file upload support
 */
public function update(Request $request, $id)
{
    DB::beginTransaction();

    try {
        // Find the existing budget entry
        $existingEntry = BudgetEntry::find($id);

        if (!$existingEntry) {
            return response()->json([
                'success' => false,
                'message' => 'Budget entry not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'Amount' => 'sometimes|numeric|min:0',
            'Sug_Amount' => 'nullable|numeric|min:0',
            'Beg_Doc' => 'nullable|array',
            'Beg_Doc.*' => 'nullable|string',
            'Beg_Doc_Names' => 'nullable|array',
            'Beg_Doc_Names.*' => 'nullable|string|max:255',
            'Beg_Doc_Types' => 'nullable|array',
            'Beg_Doc_Types.*' => 'nullable|string|max:50',
            'Status' => 'sometimes|string|in:D,R,P,F,A,I,RR,RJ,J,Draft,Revised,Approve,Freeze,Active,Inactive,Re-Revised,Reject,Justification',
            'remarks' => 'nullable|string|max:500',
            'delete_existing_docs' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        Log::info('=== BUDGET ENTRY VERSION UPDATE START ===');
        Log::info('Existing Budget Entry ID: ' . $id);
        Log::info('Existing Status: ' . $existingEntry->Status);
        Log::info('Request Status: ' . ($request->Status ?? 'NULL'));

        // Get current and requested status
        $currentStatus = $existingEntry->Status;
        $requestedStatus = $request->Status ?? null;
        $newStatus = $requestedStatus ? $this->convertToStatusCode($requestedStatus) : $currentStatus;

        Log::info('New status: ' . $newStatus);

        // Mark existing entry as not latest
        $existingEntry->update(['Is_Latest' => false]);

        // Handle document uploads
        $begDocData = $existingEntry->Beg_Doc ?? [];

        // Delete existing documents if requested
        if ($request->has('delete_existing_docs') && $request->delete_existing_docs == true) {
            $this->deleteBudgetDocuments($begDocData);
            $begDocData = [];
        }

        // Process new documents if provided
        if ($request->has('Beg_Doc') && is_array($request->Beg_Doc)) {
            $newDocs = $this->processUploadedDocuments(
                $request->Beg_Doc,
                $request->Beg_Doc_Names ?? [],
                $request->Beg_Doc_Types ?? []
            );

            // Merge with existing or replace based on request
            if ($request->has('replace_documents') && $request->replace_documents == true) {
                // Delete old documents if replacing
                $this->deleteBudgetDocuments($begDocData);
                $begDocData = $newDocs;
            } else {
                // Merge with existing
                $begDocData = array_merge($begDocData, $newDocs);
            }
        }

        // Find root entry (first version)
        $rootId = $existingEntry->Ref_Id == 0 ? $existingEntry->Budg_Id : $existingEntry->getRootEntryAttribute()->Budg_Id;

        // Create new version
        $newBudgetEntry = BudgetEntry::create([
            'Fin_Year' => $existingEntry->Fin_Year,
            'Dept_Id' => $existingEntry->Dept_Id,
            'Ledg_ID' => $existingEntry->Ledg_ID,
            'Amount' => $request->Amount ?? $existingEntry->Amount,
            'Sug_Amount' => $this->getUpdatedSugAmount($existingEntry, $request, $currentStatus, $newStatus),
            'Beg_Doc' => !empty($begDocData) ? $begDocData : ($request->Beg_Doc ?? $existingEntry->Beg_Doc),
            'Status' => $newStatus,
            'Ref_Id' => $existingEntry->Budg_Id, // Reference to previous version
            'Is_Latest' => true, // New version is always latest
        ]);

        Log::info('New Budget Entry Created - ID: ' . $newBudgetEntry->Budg_Id);
        Log::info('Reference ID: ' . $newBudgetEntry->Ref_Id);
        Log::info('Is_Latest: true');
        Log::info('Old Entry Is_Latest updated to: false');
        Log::info('Document count: ' . count($begDocData));
        Log::info('=== BUDGET ENTRY VERSION UPDATE END ===');

        DB::commit();

        $newBudgetEntry->load(['financialYear', 'department', 'ledger', 'logEntries.user', 'parentEntry']);

        return response()->json([
            'success' => true,
            'message' => 'Budget entry updated successfully (new version created)',
            'data' => $this->formatBudgetEntryData($newBudgetEntry),
            'document_info' => [
                'total_documents' => count($begDocData),
                'document_urls' => array_column($begDocData, 'url'),
            ],
            'version_info' => [
                'old_version_id' => $existingEntry->Budg_Id,
                'new_version_id' => $newBudgetEntry->Budg_Id,
                'root_id' => $rootId,
                'version_number' => $newBudgetEntry->version_number,
                'total_versions' => $newBudgetEntry->getVersionHistory()->count(),
                'status_change' => $currentStatus !== $newStatus,
                'is_approve_to_draft' => ($currentStatus === 'P' && $newStatus === 'D'),
            ]
        ]);
    } catch (\Exception $e) {
        DB::rollBack();

        Log::error('Budget entry update error: ' . $e->getMessage());

        return response()->json([
            'success' => false,
            'message' => 'Failed to update budget entry',
            'error' => $e->getMessage()
        ], 500);
    }
}

/**
 * Process uploaded documents from Base64 strings - Only store URL
 */
private function processUploadedDocuments($base64Docs, $fileNames = [], $fileTypes = [])
{
    $processedDocs = [];

    foreach ($base64Docs as $index => $base64Data) {
        try {
            // Skip if empty
            if (empty($base64Data)) {
                continue;
            }

            // Extract file info
            $fileName = $fileNames[$index] ?? 'document_' . time() . '_' . $index;
            $fileType = $fileTypes[$index] ?? $this->detectFileTypeFromBase64($base64Data);
            $fileExtension = $this->getExtensionFromType($fileType);

            // Generate unique file name
            $uniqueFileName = 'budget_doc_' . time() . '_' . uniqid() . '_' . $index . '.' . $fileExtension;

            // Define storage path (using public storage)
            $storagePath = 'budget_documents/' . date('Y') . '/' . date('m');
            $fullPath = public_path('storage/' . $storagePath . '/' . $uniqueFileName);

            // Create directory if not exists
            if (!file_exists(dirname($fullPath))) {
                mkdir(dirname($fullPath), 0777, true);
            }

            // Decode and save the file
            $fileData = $this->extractBase64Data($base64Data);
            file_put_contents($fullPath, $fileData);

            // Generate URL - Direct public URL
            $fileUrl = url('storage/' . $storagePath . '/' . $uniqueFileName);

            // Store ONLY URL and name in database
            $processedDocs[] = [
                'name' => $fileName,
                'url' => $fileUrl,
            ];

            Log::info("Document uploaded: {$fileName}, URL: {$fileUrl}");

        } catch (\Exception $e) {
            Log::error("Error processing document {$index}: " . $e->getMessage());
            continue; // Skip this document but continue with others
        }
    }

    return $processedDocs;
}



/**
 * Extract base64 data (remove data:image/png;base64, prefix)
 */
private function extractBase64Data($base64String)
{
    $base64String = trim($base64String);

    // Check for data URL format
    if (strpos($base64String, 'data:') === 0 && strpos($base64String, 'base64,') !== false) {
        $parts = explode('base64,', $base64String);
        if (count($parts) > 1) {
            $base64Data = $parts[1];
            Log::info('Extracted base64 data after base64, prefix');
            return base64_decode($base64Data, true);
        }
    }

    // Try to decode directly
    $decoded = base64_decode($base64String, true);

    if ($decoded === false) {
        Log::error('Failed to decode base64 string');
        return '';
    }

    return $decoded;
}
/**
 * Detect file type from base64 string with better debugging
 */
private function detectFileTypeFromBase64($base64String)
{
    Log::info('=== DETECTING FILE TYPE FROM BASE64 ===');
    Log::info('Base64 string length: ' . strlen($base64String));
    Log::info('First 100 chars: ' . substr($base64String, 0, 100));

    // Remove any whitespace
    $base64String = trim($base64String);

    // Check for data URL prefix
    if (preg_match('/^data:([a-zA-Z0-9]+\/[a-zA-Z0-9-.+]+);base64,/', $base64String, $matches)) {
        Log::info('Found MIME type from data URL: ' . $matches[1]);
        return $matches[1];
    } else {
        Log::info('No data URL prefix found');
    }

    // Extract and decode base64 data
    $decoded = $this->extractBase64Data($base64String);
    Log::info('Decoded data length: ' . strlen($decoded));

    if (strlen($decoded) > 0) {
        // Get first 10 bytes for signature checking
        $firstBytes = substr($decoded, 0, 10);
        Log::info('First 10 bytes (hex): ' . bin2hex($firstBytes));

        // Check for PDF signature
        if (strpos($decoded, "%PDF") === 0) {
            Log::info('Detected as PDF from signature');
            return 'application/pdf';
        }

        // Check for other signatures
        $signatures = [
            'application/pdf' => "%PDF", // PDF signature
            'image/jpeg' => "\xFF\xD8\xFF",
            'image/png' => "\x89PNG\r\n\x1A\n",
            'image/gif' => "GIF",
            'application/msword' => "\xD0\xCF\x11\xE0\xA1\xB1\x1A\xE1", // DOC
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => "PK\x03\x04", // DOCX (ZIP)
            'application/vnd.ms-excel' => "\xD0\xCF\x11\xE0\xA1\xB1\x1A\xE1", // XLS
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => "PK\x03\x04", // XLSX (ZIP)
        ];

        foreach ($signatures as $mime => $signature) {
            if (strpos($decoded, $signature) === 0) {
                Log::info("Detected as {$mime} from signature");
                return $mime;
            }
        }
    }

    Log::info('Could not detect file type, defaulting to octet-stream');
    return 'application/octet-stream';
}

/**
 * Get file extension from MIME type
 */
private function getExtensionFromType($mimeType)
{
    $mimeMap = [
        'application/pdf' => 'pdf',
        'application/msword' => 'doc',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx',
        'application/vnd.ms-excel' => 'xls',
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => 'xlsx',
        'image/jpeg' => 'jpg',
        'image/png' => 'png',
        'image/gif' => 'gif',
        'image/webp' => 'webp',
    ];

    return $mimeMap[$mimeType] ?? 'bin';
}

/**
 * Calculate base64 file size in MB
 */
private function getBase64FileSize($base64String)
{
    // Remove data URL prefix if present
    if (strpos($base64String, 'base64,') !== false) {
        $base64String = substr($base64String, strpos($base64String, ',') + 1);
    }

    $size = (strlen($base64String) * 3) / 4; // Approximate size in bytes
    return round($size / (1024 * 1024), 2); // Convert to MB
}

/**
 * Delete budget documents from storage
 */
/**
 * Delete budget documents from storage
 */
private function deleteBudgetDocuments($documents)
{
    if (!is_array($documents)) {
        return;
    }

    foreach ($documents as $doc) {
        try {
            // Extract path from URL if available
            if (isset($doc['url'])) {
                $url = $doc['url'];
                // Extract relative path from URL
                $parsedUrl = parse_url($url);
                if (isset($parsedUrl['path'])) {
                    $relativePath = ltrim($parsedUrl['path'], '/');
                    // Remove 'storage/' from path if it's in public storage
                    if (strpos($relativePath, 'storage/') === 0) {
                        $fullPath = public_path($relativePath);
                        if (file_exists($fullPath)) {
                            unlink($fullPath);
                            Log::info("Deleted document from URL: " . $fullPath);
                        }
                    }
                }
            }
            // Also check old format for backward compatibility
            elseif (isset($doc['path'])) {
                $fullPath = public_path($doc['path']);
                if (file_exists($fullPath)) {
                    unlink($fullPath);
                    Log::info("Deleted document from path: " . $fullPath);
                }
            }
        } catch (\Exception $e) {
            Log::error("Error deleting document: " . $e->getMessage());
        }
    }
}

    /**
     * Helper method to determine Sug_Amount for new version
     */
    private function getUpdatedSugAmount($existingEntry, $request, $currentStatus, $newStatus)
    {
        // If status changes from Approve to Draft, clear Sug_Amount
        if ($currentStatus === 'P' && $newStatus === 'D') {
            Log::info('APPROVE → DRAFT DETECTED - Clearing Sug_Amount');
            return null;
        }

        // If Sug_Amount is provided in request, use it
        if ($request->has('Sug_Amount')) {
            Log::info('Using Sug_Amount from request: ' . $request->Sug_Amount);
            return $request->Sug_Amount;
        }

        // Otherwise keep existing Sug_Amount
        Log::info('Keeping existing Sug_Amount: ' . $existingEntry->Sug_Amount);
        return $existingEntry->Sug_Amount;
    }

    /**
     * Get version history for a budget entry
     */
    public function getVersionHistory($budgId)
    {
        try {
            $budgetEntry = BudgetEntry::find($budgId);

            if (!$budgetEntry) {
                return response()->json([
                    'success' => false,
                    'message' => 'Budget entry not found'
                ], 404);
            }

            // Get root entry
            $rootEntry = $budgetEntry->getRootEntryAttribute();

            // Get all versions
            $versions = BudgetEntry::where('Budg_Id', $rootEntry->Budg_Id)
                ->orWhere('Ref_Id', $rootEntry->Budg_Id)
                ->with(['financialYear', 'department', 'ledger', 'logEntries.user'])
                ->orderBy('created_at', 'asc')
                ->get();

            // Calculate changes between versions
            $formattedVersions = [];
            $previousVersion = null;

            foreach ($versions as $index => $version) {
                $versionData = $this->formatBudgetEntryData($version);

                // Calculate changes from previous version
                $changes = [];
                if ($previousVersion) {
                    if ($previousVersion->Amount != $version->Amount) {
                        $changes[] = [
                            'field' => 'Amount',
                            'old' => (float) $previousVersion->Amount,
                            'new' => (float) $version->Amount,
                            'change' => (float) ($version->Amount - $previousVersion->Amount),
                            'formatted_change' => ($version->Amount > $previousVersion->Amount ? '+' : '') .
                                                 '₹ ' . number_format($version->Amount - $previousVersion->Amount, 2)
                        ];
                    }

                    if ($previousVersion->Status != $version->Status) {
                        $changes[] = [
                            'field' => 'Status',
                            'old' => $this->convertToStatusText($previousVersion->Status),
                            'new' => $this->convertToStatusText($version->Status),
                            'old_code' => $previousVersion->Status,
                            'new_code' => $version->Status
                        ];
                    }
                }

                $versionData['version_number'] = $index + 1;
                $versionData['is_latest'] = $version->is_latest;
                $versionData['ref_id'] = $version->Ref_Id;
                $versionData['changes_from_previous'] = $changes;

                $formattedVersions[] = $versionData;
                $previousVersion = $version;
            }

            return response()->json([
                'success' => true,
                'message' => 'Version history retrieved successfully',
                'root_entry_id' => $rootEntry->Budg_Id,
                'current_version_id' => $budgetEntry->Budg_Id,
                'total_versions' => $versions->count(),
                'versions' => $formattedVersions,
                'summary' => [
                    'first_version_date' => $versions->first()->created_at->format('Y-m-d H:i:s'),
                    'last_version_date' => $versions->last()->created_at->format('Y-m-d H:i:s'),
                    'time_span_days' => $versions->first()->created_at->diffInDays($versions->last()->created_at),
                    'status_transitions' => $this->getStatusTransitions($versions),
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve version history',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper to get status transitions
     */
    private function getStatusTransitions($versions)
    {
        $transitions = [];
        $previousStatus = null;

        foreach ($versions as $version) {
            if ($previousStatus !== null && $previousStatus !== $version->Status) {
                $transitions[] = [
                    'from' => $this->convertToStatusText($previousStatus),
                    'to' => $this->convertToStatusText($version->Status),
                    'from_code' => $previousStatus,
                    'to_code' => $version->Status,
                    'date' => $version->created_at->format('Y-m-d H:i:s'),
                    'version' => $version->Budg_Id
                ];
            }
            $previousStatus = $version->Status;
        }

        return $transitions;
    }

   private function formatBudgetEntryData($item)
{
    $data = [
        'budg_id' => $item->Budg_Id,
        'fin_year' => $item->Fin_Year,
        'dept_id' => $item->Dept_Id,
        'ledg_id' => $item->Ledg_ID,
        'amount' => (float) $item->Amount,
        'formatted_amount' => '₹ ' . number_format($item->Amount, 2),
        'sug_amount' => $item->Sug_Amount ? (float) $item->Sug_Amount : null,
        'formatted_sug_amount' => $item->Sug_Amount ? '₹ ' . number_format($item->Sug_Amount, 2) : null,
        'beg_doc' => $this->formatBegDocData($item->Beg_Doc),
        'document_count' => $item->document_count,
        'has_documents' => $item->has_documents,
        'status' => $item->Status,
        'status_text' => $this->convertToStatusText($item->Status),
        'is_workflow_status' => $item->is_workflow_status,
        'is_active' => $item->is_active,
        'can_edit' => $item->can_edit,
        'ref_id' => $item->Ref_Id,
        'is_latest' => (bool) $item->Is_Latest, // Changed from computed property to database field
        'created_at' => $item->created_at ? $item->created_at->format('Y-m-d H:i:s') : null,
        'updated_at' => $item->updated_at ? $item->updated_at->format('Y-m-d H:i:s') : null,
    ];

    // ... rest of the code remains same
    // Add relationships if loaded
    if ($item->relationLoaded('financialYear') && $item->financialYear) {
        $data['financial_year'] = [
            'financial_year_id' => $item->financialYear->financial_year_id,
            'financial_year_code' => $item->financialYear->financial_year_code ?? null,
            'financial_year_name' => $item->financialYear->financial_year_name ?? null,
            'start_date' => $item->financialYear->start_date ?? null,
            'end_date' => $item->financialYear->end_date ?? null,
        ];
    }

    if ($item->relationLoaded('department') && $item->department) {
        $data['department'] = [
            'id' => $item->department->id,
            'desc' => $item->department->desc,
            'isvisible' => $item->department->isvisible ?? null,
        ];
    }

    if ($item->relationLoaded('ledger') && $item->ledger) {
        $data['ledger'] = [
            'lg_id' => $item->ledger->Lg_Id,
            'lg_code' => $item->ledger->Lg_Code ?? null,
            'lg_name' => $item->ledger->Lg_Name ?? null,
            'lg_gr' => $item->ledger->Lg_Gr ?? null,
            'lg_status' => $item->ledger->Lg_Status ?? null,
            'status_text' => $item->ledger->Lg_Status ? 'Active' : 'Inactive',
        ];

        if ($item->ledger->relationLoaded('ledgerGroup') && $item->ledger->ledgerGroup) {
            $data['ledger']['ledger_group'] = [
                'lg_gr_id' => $item->ledger->ledgerGroup->Lg_Gr_Id ?? null,
                'lg_gr_name' => $item->ledger->ledgerGroup->Lg_Gr_Name ?? null,
            ];
        }
    }

    // Add parent entry info if loaded and exists
    if ($item->relationLoaded('parentEntry') && $item->parentEntry) {
        $data['parent_entry'] = $this->formatParentEntryData($item->parentEntry);
    }

    // Add child entries info if loaded
    if ($item->relationLoaded('childEntries') && $item->childEntries->isNotEmpty()) {
        $data['child_entries'] = $item->childEntries->map(function ($child) {
            return $this->formatChildEntryData($child);
        });
    }

    return $data;
}

/**
 * Format Beg_Doc data to return only name and URL
 */
private function formatBegDocData($begDoc)
{
    if (!is_array($begDoc) || empty($begDoc)) {
        return [];
    }

    $formattedDocs = [];

    foreach ($begDoc as $doc) {
        // If document has old format with extra fields, extract only name and URL
        if (is_array($doc)) {
            $formattedDocs[] = [
                'name' => $doc['name'] ?? 'document',
                'url' => $doc['url'] ?? null,
            ];
        } elseif (is_string($doc)) {
            // For backward compatibility
            $formattedDocs[] = [
                'name' => 'document',
                'url' => $doc,
            ];
        }
    }

    return $formattedDocs;
}
   /**
 * Format parent entry data (without recursive parent)
 */
private function formatParentEntryData($parentEntry)
{
    $data = [
        'budg_id' => $parentEntry->Budg_Id,
        'fin_year' => $parentEntry->Fin_Year,
        'dept_id' => $parentEntry->Dept_Id,
        'ledg_id' => $parentEntry->Ledg_ID,
        'amount' => (float) $parentEntry->Amount,
        'formatted_amount' => '₹ ' . number_format($parentEntry->Amount, 2),
        'sug_amount' => $parentEntry->Sug_Amount ? (float) $parentEntry->Sug_Amount : null,
        'formatted_sug_amount' => $parentEntry->Sug_Amount ? '₹ ' . number_format($parentEntry->Sug_Amount, 2) : null,
        'beg_doc' => $this->formatBegDocData($parentEntry->Beg_Doc), // Updated line
        'document_count' => $parentEntry->document_count,
        'has_documents' => $parentEntry->has_documents,
        'status' => $parentEntry->Status,
        'status_text' => $this->convertToStatusText($parentEntry->Status),
        'is_workflow_status' => $parentEntry->is_workflow_status,
        'is_active' => $parentEntry->is_active,
        'can_edit' => $parentEntry->can_edit,
        'ref_id' => $parentEntry->Ref_Id,
        'is_latest' => $parentEntry->is_latest,
        'created_at' => $parentEntry->created_at ? $parentEntry->created_at->format('Y-m-d H:i:s') : null,
        'updated_at' => $parentEntry->updated_at ? $parentEntry->updated_at->format('Y-m-d H:i:s') : null,
    ];

    // ... rest of the existing code remains same
    // Add relationships if loaded
    if ($parentEntry->relationLoaded('financialYear') && $parentEntry->financialYear) {
        $data['financial_year'] = [
            'financial_year_id' => $parentEntry->financialYear->financial_year_id,
            'financial_year_code' => $parentEntry->financialYear->financial_year_code ?? null,
            'financial_year_name' => $parentEntry->financialYear->financial_year_name ?? null,
            'start_date' => $parentEntry->financialYear->start_date ?? null,
            'end_date' => $parentEntry->financialYear->end_date ?? null,
        ];
    }

    if ($parentEntry->relationLoaded('department') && $parentEntry->department) {
        $data['department'] = [
            'id' => $parentEntry->department->id,
            'desc' => $parentEntry->department->desc,
            'isvisible' => $parentEntry->department->isvisible ?? null,
        ];
    }

    if ($parentEntry->relationLoaded('ledger') && $parentEntry->ledger) {
        $data['ledger'] = [
            'lg_id' => $parentEntry->ledger->Lg_Id,
            'lg_code' => $parentEntry->ledger->Lg_Code ?? null,
            'lg_name' => $parentEntry->ledger->Lg_Name ?? null,
            'lg_gr' => $parentEntry->ledger->Lg_Gr ?? null,
            'lg_status' => $parentEntry->ledger->Lg_Status ?? null,
            'status_text' => $parentEntry->ledger->Lg_Status ? 'Active' : 'Inactive',
        ];

        if ($parentEntry->ledger->relationLoaded('ledgerGroup') && $parentEntry->ledger->ledgerGroup) {
            $data['ledger']['ledger_group'] = [
                'lg_gr_id' => $parentEntry->ledger->ledgerGroup->Lg_Gr_Id ?? null,
                'lg_gr_name' => $parentEntry->ledger->ledgerGroup->Lg_Gr_Name ?? null,
            ];
        }
    }

    return $data;
}

/**
 * Format child entry data (without recursive child)
 */
private function formatChildEntryData($childEntry)
{
    $data = [
        'budg_id' => $childEntry->Budg_Id,
        'fin_year' => $childEntry->Fin_Year,
        'dept_id' => $childEntry->Dept_Id,
        'ledg_id' => $childEntry->Ledg_ID,
        'amount' => (float) $childEntry->Amount,
        'formatted_amount' => '₹ ' . number_format($childEntry->Amount, 2),
        'sug_amount' => $childEntry->Sug_Amount ? (float) $childEntry->Sug_Amount : null,
        'formatted_sug_amount' => $childEntry->Sug_Amount ? '₹ ' . number_format($childEntry->Sug_Amount, 2) : null,
        'beg_doc' => $this->formatBegDocData($childEntry->Beg_Doc), // Updated line
        'document_count' => $childEntry->document_count,
        'has_documents' => $childEntry->has_documents,
        'status' => $childEntry->Status,
        'status_text' => $this->convertToStatusText($childEntry->Status),
        'is_workflow_status' => $childEntry->is_workflow_status,
        'is_active' => $childEntry->is_active,
        'can_edit' => $childEntry->can_edit,
        'ref_id' => $childEntry->Ref_Id,
        'is_latest' => $childEntry->is_latest,
        'created_at' => $childEntry->created_at ? $childEntry->created_at->format('Y-m-d H:i:s') : null,
        'updated_at' => $childEntry->updated_at ? $childEntry->updated_at->format('Y-m-d H:i:s') : null,
    ];

    // ... rest of the existing code remains same
    // Add relationships if loaded
    if ($childEntry->relationLoaded('financialYear') && $childEntry->financialYear) {
        $data['financial_year'] = [
            'financial_year_id' => $childEntry->financialYear->financial_year_id,
            'financial_year_code' => $childEntry->financialYear->financial_year_code ?? null,
            'financial_year_name' => $childEntry->financialYear->financial_year_name ?? null,
            'start_date' => $childEntry->financialYear->start_date ?? null,
            'end_date' => $childEntry->financialYear->end_date ?? null,
        ];
    }

    if ($childEntry->relationLoaded('department') && $childEntry->department) {
        $data['department'] = [
            'id' => $childEntry->department->id,
            'desc' => $childEntry->department->desc,
            'isvisible' => $childEntry->department->isvisible ?? null,
        ];
    }

    if ($childEntry->relationLoaded('ledger') && $childEntry->ledger) {
        $data['ledger'] = [
            'lg_id' => $childEntry->ledger->Lg_Id,
            'lg_code' => $childEntry->ledger->Lg_Code ?? null,
            'lg_name' => $childEntry->ledger->Lg_Name ?? null,
            'lg_gr' => $childEntry->ledger->Lg_Gr ?? null,
            'lg_status' => $childEntry->ledger->Lg_Status ?? null,
            'status_text' => $childEntry->ledger->Lg_Status ? 'Active' : 'Inactive',
        ];

        if ($childEntry->ledger->relationLoaded('ledgerGroup') && $childEntry->ledger->ledgerGroup) {
            $data['ledger']['ledger_group'] = [
                'lg_gr_id' => $childEntry->ledger->ledgerGroup->Lg_Gr_Id ?? null,
                'lg_gr_name' => $childEntry->ledger->ledgerGroup->Lg_Gr_Name ?? null,
            ];
        }
    }

    return $data;
}
    /**
     * Get update history for a budget entry
     */
    public function getUpdateHistory($budgId)
    {
        try {
            $budgetEntry = BudgetEntry::find($budgId);

            if (!$budgetEntry) {
                return response()->json([
                    'success' => false,
                    'message' => 'Budget entry not found'
                ], 404);
            }

            // Load all log entries for this budget
            $logEntries = BudgetLogBook::with('user')
                ->where('Budg_Id', $budgId)
                ->orderBy('Date', 'desc')
                ->orderBy('BL_Id', 'desc')
                ->get();

            // Format the log entries
            $formattedLogs = $logEntries->map(function ($log) {
                return [
                    'log_id' => $log->BL_Id,
                    'status' => $log->Status,
                    'previous_status' => $log->previous_status,
                    'status_text' => $this->convertToStatusText($log->Status),
                    'previous_status_text' => $log->previous_status ? $this->convertToStatusText($log->previous_status) : null,
                    'date' => $log->Date ? $log->Date->format('Y-m-d H:i:s') : null,
                    'formatted_date' => $log->Date ? $log->Date->format('d-m-Y h:i A') : null,
                    'budget_amount' => (float) $log->budget_amount,
                    'updated_amount' => (float) $log->updated_amount,
                    'amount_change' => (float) ($log->updated_amount - $log->budget_amount),
                    'formatted_budget_amount' => '₹ ' . number_format($log->budget_amount, 2),
                    'formatted_updated_amount' => '₹ ' . number_format($log->updated_amount, 2),
                    'formatted_amount_change' => $log->updated_amount != $log->budget_amount ?
                        ($log->updated_amount > $log->budget_amount ? '+' : '') .
                        '₹ ' . number_format($log->updated_amount - $log->budget_amount, 2) : 'No change',
                    'suggested_amount' => $log->suggested_amount ? (float) $log->suggested_amount : null,
                    'formatted_suggested_amount' => $log->suggested_amount ? '₹ ' . number_format($log->suggested_amount, 2) : null,
                    'remarks' => $log->remarks,
                    'action_type' => $this->getActionType($log),
                    'user' => $log->user ? [
                        'id' => $log->user->id,
                        'name' => $log->user->name,
                        'email' => $log->user->email
                    ] : null,
                    'created_at' => $log->created_at ? $log->created_at->format('Y-m-d H:i:s') : null,
                ];
            });

            // Get summary statistics
            $totalUpdates = $logEntries->count();
            $statusChanges = $logEntries->where(function ($log) {
                return $log->previous_status !== null && $log->Status !== $log->previous_status;
            })->count();

            $amountChanges = $logEntries->where(function ($log) {
                return $log->budget_amount != $log->updated_amount;
            })->count();

            // Get first and last update
            $firstUpdate = $logEntries->last();
            $lastUpdate = $logEntries->first();

            return response()->json([
                'success' => true,
                'message' => 'Budget entry update history retrieved successfully',
                'budget_entry' => [
                    'budg_id' => $budgetEntry->Budg_Id,
                    'fin_year' => $budgetEntry->Fin_Year,
                    'dept_id' => $budgetEntry->Dept_Id,
                    'ledg_id' => $budgetEntry->Ledg_ID,
                    'current_amount' => (float) $budgetEntry->Amount,
                    'formatted_current_amount' => '₹ ' . number_format($budgetEntry->Amount, 2),
                    'current_sug_amount' => $budgetEntry->Sug_Amount ? (float) $budgetEntry->Sug_Amount : null,
                    'formatted_current_sug_amount' => $budgetEntry->Sug_Amount ? '₹ ' . number_format($budgetEntry->Sug_Amount, 2) : null,
                    'current_status' => $budgetEntry->Status,
                    'current_status_text' => $this->convertToStatusText($budgetEntry->Status),
                    'document_count' => $budgetEntry->document_count,
                    'has_documents' => $budgetEntry->has_documents,
                    'ref_id' => $budgetEntry->Ref_Id,
                    'is_latest' => $budgetEntry->is_latest,
                    'version_number' => $budgetEntry->version_number,
                    'created_at' => $budgetEntry->created_at ? $budgetEntry->created_at->format('Y-m-d H:i:s') : null,
                    'last_updated_at' => $budgetEntry->updated_at ? $budgetEntry->updated_at->format('Y-m-d H:i:s') : null,
                ],
                'update_history' => $formattedLogs,
                'summary' => [
                    'total_updates' => $totalUpdates,
                    'status_changes' => $statusChanges,
                    'amount_changes' => $amountChanges,
                    'first_update' => $firstUpdate ? [
                        'date' => $firstUpdate->Date ? $firstUpdate->Date->format('Y-m-d H:i:s') : null,
                        'status' => $firstUpdate->Status,
                        'status_text' => $this->convertToStatusText($firstUpdate->Status),
                        'amount' => (float) $firstUpdate->updated_amount,
                        'formatted_amount' => '₹ ' . number_format($firstUpdate->updated_amount, 2),
                    ] : null,
                    'last_update' => $lastUpdate ? [
                        'date' => $lastUpdate->Date ? $lastUpdate->Date->format('Y-m-d H:i:s') : null,
                        'status' => $lastUpdate->Status,
                        'status_text' => $this->convertToStatusText($lastUpdate->Status),
                        'amount' => (float) $lastUpdate->updated_amount,
                        'formatted_amount' => '₹ ' . number_format($lastUpdate->updated_amount, 2),
                    ] : null,
                    'timeline' => [
                        'total_days' => $firstUpdate && $lastUpdate ?
                            $lastUpdate->Date->diffInDays($firstUpdate->Date) : 0,
                        'updates_per_day' => $totalUpdates > 0 && $firstUpdate && $lastUpdate &&
                            $lastUpdate->Date->diffInDays($firstUpdate->Date) > 0 ?
                            round($totalUpdates / $lastUpdate->Date->diffInDays($firstUpdate->Date), 2) : 0,
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve budget entry update history',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper method to determine action type
     */
    private function getActionType($log)
    {
        if (!$log->previous_status) {
            return 'Created';
        }

        if ($log->Status !== $log->previous_status) {
            return 'Status Change';
        }

        if ($log->budget_amount != $log->updated_amount) {
            return 'Amount Update';
        }

        if ($log->suggested_amount != null) {
            return 'Suggestion Added/Updated';
        }

        return 'Updated';
    }
}

