<?php

namespace App\Http\Controllers;

use App\Models\BudgetLogBook;
use App\Models\BudgetEntry;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class BudgetLogBookController extends Controller
{
    /**
     * Display a listing of budget log entries.
     */
    public function index(Request $request)
    {
        try {
            $query = BudgetLogBook::query();

            // Apply filters
            if ($request->has('budg_id')) {
                $query->where('Budg_Id', $request->budg_id);
            }

            if ($request->has('status')) {
                $query->where('Status', $request->status);
            }

            if ($request->has('date_from')) {
                $query->whereDate('Date', '>=', $request->date_from);
            }

            if ($request->has('date_to')) {
                $query->whereDate('Date', '<=', $request->date_to);
            }

            if ($request->has('user_id')) {
                $query->where('user_id', $request->user_id);
            }

            // Load relationships
            $query->with(['budgetEntry', 'user']);

            // Sorting
            $sortField = $request->get('sort_by', 'Date');
            $sortOrder = $request->get('sort_order', 'desc');
            $query->orderBy($sortField, $sortOrder);

            // Pagination
            $perPage = $request->get('per_page', 20);
            $logEntries = $query->paginate($perPage);

            // Format response data
            $formattedData = $logEntries->map(function ($item) {
                return $this->formatLogEntryData($item);
            });

            // Calculate summary
            $totalEntries = $logEntries->total();
            $statusSummary = BudgetLogBook::select('Status', DB::raw('COUNT(*) as count'))
                ->groupBy('Status')
                ->get()
                ->pluck('count', 'Status')
                ->toArray();

            return response()->json([
                'success' => true,
                'message' => 'Budget log entries retrieved successfully',
                'data' => $formattedData,
                'summary' => [
                    'total_logs' => $totalEntries,
                    'status_summary' => $statusSummary,
                ],
                'pagination' => [
                    'current_page' => $logEntries->currentPage(),
                    'per_page' => $logEntries->perPage(),
                    'total' => $logEntries->total(),
                    'last_page' => $logEntries->lastPage(),
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve budget log entries',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified budget log entry.
     */
    public function show($id)
    {
        try {
            $logEntry = BudgetLogBook::with(['budgetEntry', 'user'])->find($id);

            if (!$logEntry) {
                return response()->json([
                    'success' => false,
                    'message' => 'Budget log entry not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Budget log entry retrieved successfully',
                'data' => $this->formatLogEntryData($logEntry)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve budget log entry',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get budget logs by budget entry ID
     */
    public function getByBudgetEntry($budgId)
    {
        try {
            $budgetEntry = BudgetEntry::find($budgId);

            if (!$budgetEntry) {
                return response()->json([
                    'success' => false,
                    'message' => 'Budget entry not found'
                ], 404);
            }

            $logEntries = BudgetLogBook::with(['user'])
                ->where('Budg_Id', $budgId)
                ->orderBy('Date', 'desc')
                ->orderBy('BL_Id', 'desc')
                ->get();

            $formattedData = $logEntries->map(function ($item) {
                return $this->formatLogEntryData($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Budget logs retrieved successfully',
                'data' => $formattedData,
                'summary' => [
                    'budg_id' => $budgId,
                    'total_logs' => $logEntries->count(),
                    'current_status' => $budgetEntry->Status,
                    'current_amount' => (float) $budgetEntry->Amount,
                    'formatted_current_amount' => '₹ ' . number_format($budgetEntry->Amount, 2)
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve budget logs',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get budget logs by status
     */
    public function getByStatus($status)
    {
        try {
            $validStatuses = ['Draft', 'Revised', 'Approve', 'Freeze', 'A', 'I'];

            if (!in_array($status, $validStatuses)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid status. Valid statuses are: ' . implode(', ', $validStatuses)
                ], 400);
            }

            $logEntries = BudgetLogBook::with(['budgetEntry', 'user'])
                ->where('Status', $status)
                ->orderBy('Date', 'desc')
                ->get();

            $formattedData = $logEntries->map(function ($item) {
                return $this->formatLogEntryData($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Budget logs retrieved successfully by status',
                'data' => $formattedData,
                'summary' => [
                    'status' => $status,
                    'total_logs' => $logEntries->count(),
                    'status_text' => $this->getWorkflowStatusText($status)
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve budget logs by status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get budget logs by date range
     */
    public function getByDateRange(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'date_from' => 'required|date',
                'date_to' => 'required|date|after_or_equal:date_from',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $logEntries = BudgetLogBook::with(['budgetEntry', 'user'])
                ->whereBetween('Date', [$request->date_from, $request->date_to])
                ->orderBy('Date', 'desc')
                ->get();

            $formattedData = $logEntries->map(function ($item) {
                return $this->formatLogEntryData($item);
            });

            // Calculate daily summary
            $dailySummary = BudgetLogBook::select(
                    DB::raw('DATE(Date) as date'),
                    DB::raw('COUNT(*) as total_logs'),
                    DB::raw('SUM(updated_amount) as total_amount'),
                    DB::raw('AVG(updated_amount) as average_amount')
                )
                ->whereBetween('Date', [$request->date_from, $request->date_to])
                ->groupBy(DB::raw('DATE(Date)'))
                ->orderBy('date', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'Budget logs retrieved successfully by date range',
                'data' => $formattedData,
                'summary' => [
                    'date_from' => $request->date_from,
                    'date_to' => $request->date_to,
                    'total_logs' => $logEntries->count(),
                    'total_amount' => (float) $logEntries->sum('updated_amount'),
                    'formatted_total_amount' => '₹ ' . number_format($logEntries->sum('updated_amount'), 2),
                    'daily_summary' => $dailySummary
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve budget logs by date range',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get budget logs by user
     */
    public function getByUser($userId)
    {
        try {
            $logEntries = BudgetLogBook::with(['budgetEntry'])
                ->where('user_id', $userId)
                ->orderBy('Date', 'desc')
                ->get();

            $formattedData = $logEntries->map(function ($item) {
                return $this->formatLogEntryData($item);
            });

            // Get user info
            $user = \App\Models\GmcUser::find($userId);

            return response()->json([
                'success' => true,
                'message' => 'Budget logs retrieved successfully by user',
                'data' => $formattedData,
                'summary' => [
                    'user_id' => $userId,
                    'user_name' => $user ? $user->name : 'Unknown User',
                    'user_email' => $user ? $user->email : null,
                    'total_logs' => $logEntries->count(),
                    'total_actions' => $logEntries->count(),
                    'first_action' => $logEntries->last() ? $logEntries->last()->Date->format('Y-m-d H:i:s') : null,
                    'last_action' => $logEntries->first() ? $logEntries->first()->Date->format('Y-m-d H:i:s') : null
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve budget logs by user',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get budget logs summary
     */
    public function getSummary(Request $request)
    {
        try {
            $query = BudgetLogBook::query();

            // Apply filters if provided
            if ($request->has('date_from')) {
                $query->whereDate('Date', '>=', $request->date_from);
            }

            if ($request->has('date_to')) {
                $query->whereDate('Date', '<=', $request->date_to);
            }

            if ($request->has('budg_id')) {
                $query->where('Budg_Id', $request->budg_id);
            }

            // Get overall summary
            $overallSummary = $query->select([
                DB::raw('COUNT(*) as total_logs'),
                DB::raw('COUNT(DISTINCT Budg_Id) as unique_budget_entries'),
                DB::raw('COUNT(DISTINCT user_id) as unique_users'),
                DB::raw('SUM(updated_amount) as total_amount'),
                DB::raw('AVG(updated_amount) as average_amount'),
                DB::raw('MAX(updated_amount) as max_amount'),
                DB::raw('MIN(updated_amount) as min_amount'),
            ])->first();

            // Get status-wise summary
            $statusSummary = BudgetLogBook::select([
                'Status',
                DB::raw('COUNT(*) as count'),
                DB::raw('SUM(updated_amount) as total_amount'),
                DB::raw('AVG(updated_amount) as average_amount'),
            ])
            ->groupBy('Status')
            ->get()
            ->map(function ($item) use ($overallSummary) {
                return [
                    'status' => $item->Status,
                    'status_text' => $this->getWorkflowStatusText($item->Status),
                    'count' => $item->count,
                    'total_amount' => (float) $item->total_amount,
                    'formatted_total_amount' => '₹ ' . number_format($item->total_amount, 2),
                    'average_amount' => (float) $item->average_amount,
                    'percentage' => $overallSummary->total_logs > 0 ?
                        round(($item->count / $overallSummary->total_logs) * 100, 2) : 0
                ];
            });

            // Get daily activity
            $dailyActivity = BudgetLogBook::select([
                DB::raw('DATE(Date) as date'),
                DB::raw('COUNT(*) as log_count'),
                DB::raw('COUNT(DISTINCT Budg_Id) as budget_entry_count'),
                DB::raw('COUNT(DISTINCT user_id) as user_count'),
            ])
            ->groupBy(DB::raw('DATE(Date)'))
            ->orderBy('date', 'desc')
            ->limit(30)
            ->get();

            // Get top users
            $topUsers = BudgetLogBook::select([
                'user_id',
                DB::raw('COUNT(*) as action_count'),
                DB::raw('COUNT(DISTINCT Budg_Id) as budget_entry_count'),
            ])
            ->whereNotNull('user_id')
            ->groupBy('user_id')
            ->orderBy('action_count', 'desc')
            ->limit(10)
            ->with('user')
            ->get()
            ->map(function ($item) {
                return [
                    'user_id' => $item->user_id,
                    'user_name' => $item->user ? $item->user->name : 'Unknown',
                    'user_email' => $item->user ? $item->user->email : null,
                    'action_count' => $item->action_count,
                    'budget_entry_count' => $item->budget_entry_count,
                ];
            });

            return response()->json([
                'success' => true,
                'message' => 'Budget logs summary retrieved successfully',
                'data' => [
                    'overall_summary' => [
                        'total_logs' => (int) $overallSummary->total_logs,
                        'unique_budget_entries' => (int) $overallSummary->unique_budget_entries,
                        'unique_users' => (int) $overallSummary->unique_users,
                        'total_amount' => (float) $overallSummary->total_amount,
                        'formatted_total_amount' => '₹ ' . number_format($overallSummary->total_amount, 2),
                        'average_amount' => (float) $overallSummary->average_amount,
                        'max_amount' => (float) $overallSummary->max_amount,
                        'min_amount' => (float) $overallSummary->min_amount,
                    ],
                    'status_summary' => $statusSummary,
                    'daily_activity' => $dailyActivity,
                    'top_users' => $topUsers,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve budget logs summary',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper method to format log entry data
     */
    private function formatLogEntryData($item)
    {
        return [
            'bl_id' => $item->BL_Id,
            'budg_id' => $item->Budg_Id,
            'status' => $item->Status,
            'formatted_status' => $this->getWorkflowStatusText($item->Status),
            'previous_status' => $item->previous_status,
            'formatted_previous_status' => $item->previous_status ? $this->getWorkflowStatusText($item->previous_status) : null,
            'date' => $item->Date ? $item->Date->format('Y-m-d H:i:s') : null,
            'formatted_date' => $item->Date ? $item->Date->format('d-m-Y h:i A') : null,
            'user' => $item->user ? [
                'id' => $item->user->id,
                'name' => $item->user->name,
                'email' => $item->user->email
            ] : null,
            'budget_amount' => (float) $item->budget_amount,
            'updated_amount' => (float) $item->updated_amount,
            'amount_change' => (float) ($item->updated_amount - $item->budget_amount),
            'formatted_budget_amount' => '₹ ' . number_format($item->budget_amount, 2),
            'formatted_updated_amount' => '₹ ' . number_format($item->updated_amount, 2),
            'formatted_amount_change' => $item->updated_amount != $item->budget_amount ?
                ($item->updated_amount > $item->budget_amount ? '+' : '') .
                '₹ ' . number_format($item->updated_amount - $item->budget_amount, 2) : 'No change',
            'remarks' => $item->remarks,
            'created_at' => $item->created_at ? $item->created_at->format('Y-m-d H:i:s') : null,
            'updated_at' => $item->updated_at ? $item->updated_at->format('Y-m-d H:i:s') : null,
            'budget_entry' => $item->budgetEntry ? [
                'budg_id' => $item->budgetEntry->Budg_Id,
                'fin_year' => $item->budgetEntry->Fin_Year,
                'dept_id' => $item->budgetEntry->Dept_Id,
                'ledg_id' => $item->budgetEntry->Ledg_ID,
                'amount' => (float) $item->budgetEntry->Amount,
                'formatted_amount' => '₹ ' . number_format($item->budgetEntry->Amount, 2),
                'status' => $item->budgetEntry->Status,
                'status_text' => $item->budgetEntry->Status == 'A' ? 'Active' :
                    ($item->budgetEntry->Status == 'I' ? 'Inactive' : $this->getWorkflowStatusText($item->budgetEntry->Status)),
            ] : null,
        ];
    }

    /**
     * Get workflow status text
     */
    private function getWorkflowStatusText($status)
    {
        $statusMap = [
            'Draft' => 'Draft',
            'Revised' => 'Revised',
            'Approve' => 'Approved',
            'Freeze' => 'Frozen',
            'A' => 'Active',
            'I' => 'Inactive'
        ];

        return $statusMap[$status] ?? $status;
    }
}
