<?php

namespace App\Http\Controllers;

use App\Models\DesPostBook;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class DesPostBookController extends Controller
{
    // =====================
    // PAGINATED LIST
    // =====================
    public function index()
    {
        $data = DesPostBook::with('designationMenu')
            ->orderBy('dp_id', 'desc')
            ->paginate(15);

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

    // =====================
    // ALL DATA (NO PAGINATION)
    // =====================
    public function all()
    {
        $data = DesPostBook::with('designationMenu')
            ->orderBy('dp_id', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'total' => $data->count(),
            'data' => $data
        ]);
    }

    // =====================
    // STORE
    // =====================
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'designation_id' => 'nullable|exists:designation_menus,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = DesPostBook::create([
            'dp_date'        => $request->dp_date,
            'designation_id' => $request->designation_id,
            'per_post'       => $request->per_post,
            'temp_post'      => $request->temp_post
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Designation Post Entry created',
            'data' => $data
        ], 201);
    }

    // =====================
    // SHOW
    // =====================
    public function show($id)
    {
        $data = DesPostBook::with('designationMenu')->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

    // =====================
    // UPDATE (sometimes)
    // =====================
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'designation_id' => 'sometimes|nullable|exists:designation_menus,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = DesPostBook::findOrFail($id);

        $data->update($request->only([
            'dp_date',
            'designation_id',
            'per_post',
            'temp_post'
        ]));

        return response()->json([
            'success' => true,
            'message' => 'Designation Post Entry updated',
            'data' => $data
        ]);
    }

    // =====================
    // DELETE
    // =====================
    public function destroy($id)
    {
        DesPostBook::findOrFail($id)->delete();

        return response()->json([
            'success' => true,
            'message' => 'Designation Post Entry deleted'
        ]);
    }
}
