<?php

namespace App\Http\Controllers;

use App\Models\DesignationMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class DesignationMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $designations = DesignationMaster::with('payMappings')->get();
        return response()->json([
            'success' => true,
            'data' => $designations
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'designation_name' => 'required|string|max:100',
            'department_name' => 'required|string|max:100'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $designation = DesignationMaster::create($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Designation created successfully',
            'data' => $designation
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $designation = DesignationMaster::with(['payMappings.commission',
                                              'payMappings.payScale',
                                              'payMappings.payBand',
                                              'payMappings.gradePay',
                                              'payMappings.level'])->find($id);

        if (!$designation) {
            return response()->json([
                'success' => false,
                'message' => 'Designation not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $designation
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $designation = DesignationMaster::find($id);

        if (!$designation) {
            return response()->json([
                'success' => false,
                'message' => 'Designation not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'designation_name' => 'string|max:100',
            'department_name' => 'string|max:100'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $designation->update($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Designation updated successfully',
            'data' => $designation
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $designation = DesignationMaster::find($id);

        if (!$designation) {
            return response()->json([
                'success' => false,
                'message' => 'Designation not found'
            ], 404);
        }

        $designation->delete();

        return response()->json([
            'success' => true,
            'message' => 'Designation deleted successfully'
        ]);
    }

    /**
     * Get pay history for a designation
     */
    public function getPayHistory(string $id)
    {
        $designation = DesignationMaster::with(['payMappings.commission',
                                              'payMappings.structureType',
                                              'payMappings.payScale',
                                              'payMappings.payBand',
                                              'payMappings.gradePay',
                                              'payMappings.level'])->find($id);

        if (!$designation) {
            return response()->json([
                'success' => false,
                'message' => 'Designation not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'designation' => $designation->only(['designation_id', 'designation_name', 'department_name']),
            'pay_history' => $designation->payMappings
        ]);
    }

    /**
     * Get designations by department
     */
    public function getByDepartment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'department_name' => 'required|string|max:100'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $designations = DesignationMaster::where('department_name', $request->department_name)
            ->with('payMappings')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $designations
        ]);
    }
}
