<?php

namespace App\Http\Controllers;

use App\Models\DesignationMenu;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use League\Config\Exception\ValidationException;
use Throwable;

class DesignationMenuController extends Controller
{
    // GET ALL
    public function index(Request $request)
    {
        try {
            $queryValue = $request->query('query');
            $isvisible  = $request->query('isvisible');

            $menus = DesignationMenu::with('parentMenu');

            if ($queryValue) {
                $menus->where('Upgrp', 'like', $queryValue . '%');
            }

            if ($isvisible !== null) {
                $menus->where('isvisible', $isvisible);
            }

            return response()->json([
                'status' => 'success',
                'data'   => $menus->get()
            ]);
        } catch (Throwable $e) {
            Log::error($e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch designation menus'
            ], 500);
        }
    }

    // GET ONE
    public function show($id)
    {
        $menu = DesignationMenu::with('parentMenu')->find($id);

        if (!$menu) {
            return response()->json([
                'status' => 'error',
                'message' => 'Designation menu not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data'   => $menu
        ]);
    }

    // CREATE
    public function store(Request $request)
    {
        try {
            $grsh = $request->input('Grsh', '1');
            $grsrn = DesignationMenu::where('Grsh', $grsh)->count() + 1;

            $prefix = str_pad($grsrn, 2, '0', STR_PAD_LEFT);
            $code   = $prefix . '000000';

            $menu = DesignationMenu::create([
                'title'       => $request->title,
                'desc'        => $request->desc,
                // 'category_id' => $request->category_id,
                'func_id'     => $request->func_id,
                'Grsh'        => $grsh,
                'Grsrn'       => (string) $grsrn,
                'Upgrp'       => $request->Upgrp ?? $code,
                'Grcod'       => $request->Grcod ?? $code,
                'Menu_Links'  => $request->Menu_Links,
                'isvisible'   => $request->isvisible ?? 1,
            ]);

            return response()->json([
                'status' => 'success',
                'data'   => $menu
            ], 201);
        } catch (Throwable $e) {
            Log::error($e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to create designation menu'
            ], 500);
        }
    }

    // UPDATE
    public function update(Request $request, $id)
    {
        try {
            $menu = DesignationMenu::find($id);

            if (!$menu) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Designation menu not found'
                ], 404);
            }

            $data = $request->validate([
                'title'         => 'sometimes|nullable|string',
                'desc'          => 'sometimes|nullable|string',

                // 'category_id'   => 'sometimes|nullable|exists:categories,category_id',
                'func_id'       => 'sometimes|nullable|exists:function_types,id',
                'Menu_Links'    => 'sometimes|nullable|string',
                'Upgrp'         => 'sometimes|nullable|string',
                'Grcod'         => 'sometimes|nullable|string',
                'Grsrn'         => 'sometimes|nullable|string',
                'Grsh'          => 'sometimes|nullable|string',
                'isvisible'     => 'sometimes|nullable|integer|in:0,1',
            ]);

            $menu->update($data);

            return response()->json([
                'status'  => 'success',
                'message' => 'Designation menu updated successfully',
                'data'    => $menu
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation failed',
                'errors'  => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('DesignationMenu Update Error: ' . $e->getMessage());

            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to update designation menu',
                'error'   => $e->getMessage()
            ], 500);
        }
    }


    // TOGGLE VISIBILITY


    public function toggleVisibility($id)
    {
        try {
            $menu = DesignationMenu::find($id);

            if (!$menu) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Designation menu not found'
                ], 404);
            }

            $menu->isvisible = $menu->isvisible == 1 ? 0 : 1;
            $menu->save();

            return response()->json([
                'status'  => 'success',
                'message' => 'Visibility toggled successfully',
                'data'    => $menu
            ]);
        } catch (Throwable $e) {
            Log::error('DesignationMenu Toggle Error: ' . $e->getMessage());

            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to toggle visibility',
                'error'   => $e->getMessage()
            ], 500);
        }
    }


    // DELETE
    public function destroy($id)
    {
        $menu = DesignationMenu::find($id);

        if (!$menu) {
            return response()->json([
                'status' => 'error',
                'message' => 'Designation menu not found'
            ], 404);
        }

        $menu->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Designation menu deleted successfully'
        ]);
    }
}
