<?php

namespace App\Http\Controllers;

use App\Models\DesignationPayMapping;
use App\Models\DesignationMaster;
use App\Models\PayCommissionMaster;
use App\Models\PayStructureTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class DesignationPayMappingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $mappings = DesignationPayMapping::with([
            'designation',
            'commission',
            'structureType',
            'payScale',
            'payBand',
            'gradePay',
            'level'
        ])->get();

        return response()->json([
            'success' => true,
            'data' => $mappings
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'designation_id' => 'required|exists:designation_masters,designation_id',
            'commission_id' => 'nullable|exists:pay_commission_masters,commission_id',
            'structure_type_id' => 'nullable|exists:pay_structure_type_masters,structure_type_id',
            'pay_scale_id' => 'nullable|exists:pay_scale_masters,pay_scale_id',
            'pay_band_id' => 'nullable|exists:pay_band_masters,pay_band_id',
            'grade_pay_id' => 'nullable|exists:grade_pay_masters,grade_pay_id',
            'level_id' => 'nullable|exists:pay_matrix_level_masters,level_id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // Validate based on structure type
        $structureTypeId = $request->structure_type_id;

        if ($structureTypeId == 1) { // SCALE
            if (!$request->pay_scale_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pay Scale is required for SCALE structure type'
                ], 422);
            }
        } elseif ($structureTypeId == 2) { // PAY_BAND
            if (!$request->pay_band_id || !$request->grade_pay_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pay Band and Grade Pay are required for PAY_BAND structure type'
                ], 422);
            }
        } elseif ($structureTypeId == 3) { // PAY_MATRIX
            if (!$request->level_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Level is required for PAY_MATRIX structure type'
                ], 422);
            }
        }

        // Check if mapping already exists for this designation and commission
        $existingMapping = DesignationPayMapping::where('designation_id', $request->designation_id)
            ->where('commission_id', $request->commission_id)
            ->first();

        if ($existingMapping) {
            return response()->json([
                'success' => false,
                'message' => 'Pay mapping already exists for this designation and commission'
            ], 409);
        }

        $mapping = DesignationPayMapping::create($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Designation Pay Mapping created successfully',
            'data' => $mapping->load(['designation', 'commission', 'structureType'])
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $mapping = DesignationPayMapping::with([
            'designation',
            'commission',
            'structureType',
            'payScale',
            'payBand',
            'gradePay',
            'level'
        ])->find($id);

        if (!$mapping) {
            return response()->json([
                'success' => false,
                'message' => 'Designation Pay Mapping not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $mapping
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $mapping = DesignationPayMapping::find($id);

        if (!$mapping) {
            return response()->json([
                'success' => false,
                'message' => 'Designation Pay Mapping not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'designation_id' => 'exists:designation_masters,designation_id',
            'commission_id' => 'nullable|exists:pay_commission_masters,commission_id',
            'structure_type_id' => 'nullable|exists:pay_structure_type_masters,structure_type_id',
            'pay_scale_id' => 'nullable|exists:pay_scale_masters,pay_scale_id',
            'pay_band_id' => 'nullable|exists:pay_band_masters,pay_band_id',
            'grade_pay_id' => 'nullable|exists:grade_pay_masters,grade_pay_id',
            'level_id' => 'nullable|exists:pay_matrix_level_masters,level_id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // Check for duplicate mapping if designation or commission is being changed
        if ($request->has('designation_id') || $request->has('commission_id')) {
            $designationId = $request->designation_id ?? $mapping->designation_id;
            $commissionId = $request->commission_id ?? $mapping->commission_id;

            $existingMapping = DesignationPayMapping::where('designation_id', $designationId)
                ->where('commission_id', $commissionId)
                ->where('mapping_id', '!=', $id)
                ->first();

            if ($existingMapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pay mapping already exists for this designation and commission'
                ], 409);
            }
        }

        $mapping->update($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Designation Pay Mapping updated successfully',
            'data' => $mapping->load(['designation', 'commission', 'structureType'])
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $mapping = DesignationPayMapping::find($id);

        if (!$mapping) {
            return response()->json([
                'success' => false,
                'message' => 'Designation Pay Mapping not found'
            ], 404);
        }

        $mapping->delete();

        return response()->json([
            'success' => true,
            'message' => 'Designation Pay Mapping deleted successfully'
        ]);
    }

    /**
     * Get mappings by designation
     */
    public function getByDesignation(string $designationId)
    {
        $designation = DesignationMaster::find($designationId);

        if (!$designation) {
            return response()->json([
                'success' => false,
                'message' => 'Designation not found'
            ], 404);
        }

        $mappings = DesignationPayMapping::where('designation_id', $designationId)
            ->with(['commission', 'structureType', 'payScale', 'payBand', 'gradePay', 'level'])
            ->get();

        return response()->json([
            'success' => true,
            'data' => $mappings
        ]);
    }

    /**
     * Get mappings by commission
     */
    public function getByCommission(string $commissionId)
    {
        $commission = PayCommissionMaster::find($commissionId);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Commission not found'
            ], 404);
        }

        $mappings = DesignationPayMapping::where('commission_id', $commissionId)
            ->with(['designation', 'structureType', 'payScale', 'payBand', 'gradePay', 'level'])
            ->get();

        return response()->json([
            'success' => true,
            'data' => $mappings
        ]);
    }
}
