<?php

namespace App\Http\Controllers;

use App\Models\EMDBGTypeMaster;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class EMDBGTypeMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            // Get query parameters
            $perPage = $request->get('per_page', 10);
            $search = $request->get('search', '');
            $withTrashed = $request->get('with_trashed', false);

            // Start query
            $query = EMDBGTypeMaster::query();

            // Include soft deleted records if requested
            if ($withTrashed) {
                $query->withTrashed();
            }

            // Apply search filter
            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('emdbg_type_code', 'like', "%{$search}%")
                      ->orWhere('emdbg_type_name', 'like', "%{$search}%")
                      ->orWhere('percentage', 'like', "%{$search}%");
                });
            }

            // Apply ordering
            $query->orderBy('emdbg_id', 'desc');

            // Get paginated results
            $emdbgTypes = $query->paginate($perPage);

            // Transform the data
            $transformedData = $emdbgTypes->getCollection()->map(function ($item) {
                return $this->transformEMDBGType($item);
            });

            $emdbgTypes->setCollection($transformedData);

            return response()->json([
                'success' => true,
                'message' => 'EMDBG Type Masters retrieved successfully.',
                'data' => $emdbgTypes->items(),
                'meta' => [
                    'current_page' => $emdbgTypes->currentPage(),
                    'last_page' => $emdbgTypes->lastPage(),
                    'per_page' => $emdbgTypes->perPage(),
                    'total' => $emdbgTypes->total(),
                    'from' => $emdbgTypes->firstItem(),
                    'to' => $emdbgTypes->lastItem(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve EMDBG Type Masters.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        // Validate request
        $validator = Validator::make($request->all(), [
            'emdbg_type_code' => 'nullable|string|max:20',
            'emdbg_type_name' => 'required|string|max:255',
            'percentage' => 'nullable|integer|min:0|max:100'
        ], [
            'emdbg_type_name.required' => 'EMDBG Type Name is required.',
            'emdbg_type_code.max' => 'EMDBG Type Code must not exceed 20 characters.',
            'emdbg_type_name.max' => 'EMDBG Type Name must not exceed 255 characters.',
            'percentage.integer' => 'Percentage must be an integer.',
            'percentage.min' => 'Percentage must be at least 0.',
            'percentage.max' => 'Percentage must not exceed 100.'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Check for duplicate type code if provided
            if ($request->emdbg_type_code) {
                $existing = EMDBGTypeMaster::where('emdbg_type_code', $request->emdbg_type_code)->first();
                if ($existing) {
                    return response()->json([
                        'success' => false,
                        'message' => 'EMDBG Type Code already exists.',
                        'data' => $this->transformEMDBGType($existing)
                    ], 409);
                }
            }

            // Check for duplicate type name
            $existingName = EMDBGTypeMaster::where('emdbg_type_name', $request->emdbg_type_name)->first();
            if ($existingName) {
                return response()->json([
                    'success' => false,
                    'message' => 'EMDBG Type Name already exists.',
                    'data' => $this->transformEMDBGType($existingName)
                ], 409);
            }

            // Create new EMDBG type
            $emdbgType = EMDBGTypeMaster::create([
                'emdbg_type_code' => $request->emdbg_type_code,
                'emdbg_type_name' => $request->emdbg_type_name,
                'percentage' => $request->percentage
            ]);

            return response()->json([
                'success' => true,
                'message' => 'EMDBG Type Master created successfully.',
                'data' => $this->transformEMDBGType($emdbgType)
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create EMDBG Type Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): JsonResponse
    {
        try {
            $emdbgType = EMDBGTypeMaster::withTrashed()->find($id);

            if (!$emdbgType) {
                return response()->json([
                    'success' => false,
                    'message' => 'EMDBG Type Master not found.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'EMDBG Type Master retrieved successfully.',
                'data' => $this->transformEMDBGType($emdbgType)
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve EMDBG Type Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $emdbgType = EMDBGTypeMaster::find($id);

            if (!$emdbgType) {
                return response()->json([
                    'success' => false,
                    'message' => 'EMDBG Type Master not found.'
                ], 404);
            }

            // Validate request
            $validator = Validator::make($request->all(), [
                'emdbg_type_code' => 'nullable|string|max:20',
                'emdbg_type_name' => 'sometimes|required|string|max:255',
                'percentage' => 'sometimes|nullable|integer|min:0|max:100'
            ], [
                'emdbg_type_name.required' => 'EMDBG Type Name is required.',
                'emdbg_type_code.max' => 'EMDBG Type Code must not exceed 20 characters.',
                'emdbg_type_name.max' => 'EMDBG Type Name must not exceed 255 characters.',
                'percentage.integer' => 'Percentage must be an integer.',
                'percentage.min' => 'Percentage must be at least 0.',
                'percentage.max' => 'Percentage must not exceed 100.'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Check for duplicate type code if provided
            if ($request->has('emdbg_type_code') && $request->emdbg_type_code) {
                $existing = EMDBGTypeMaster::where('emdbg_type_code', $request->emdbg_type_code)
                    ->where('emdbg_id', '!=', $id)
                    ->first();
                if ($existing) {
                    return response()->json([
                        'success' => false,
                        'message' => 'EMDBG Type Code already exists.',
                        'data' => $this->transformEMDBGType($existing)
                    ], 409);
                }
            }

            // Check for duplicate type name if provided
            if ($request->has('emdbg_type_name')) {
                $existingName = EMDBGTypeMaster::where('emdbg_type_name', $request->emdbg_type_name)
                    ->where('emdbg_id', '!=', $id)
                    ->first();
                if ($existingName) {
                    return response()->json([
                        'success' => false,
                        'message' => 'EMDBG Type Name already exists.',
                        'data' => $this->transformEMDBGType($existingName)
                    ], 409);
                }
            }

            // Update only provided fields
            $updateData = [];
            if ($request->has('emdbg_type_code')) {
                $updateData['emdbg_type_code'] = $request->emdbg_type_code;
            }
            if ($request->has('emdbg_type_name')) {
                $updateData['emdbg_type_name'] = $request->emdbg_type_name;
            }
            if ($request->has('percentage')) {
                $updateData['percentage'] = $request->percentage;
            }

            $emdbgType->update($updateData);

            // Refresh the model to get updated data
            $emdbgType->refresh();

            return response()->json([
                'success' => true,
                'message' => 'EMDBG Type Master updated successfully.',
                'data' => $this->transformEMDBGType($emdbgType)
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update EMDBG Type Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $emdbgType = EMDBGTypeMaster::find($id);

            if (!$emdbgType) {
                return response()->json([
                    'success' => false,
                    'message' => 'EMDBG Type Master not found.'
                ], 404);
            }

            $emdbgType->delete();

            return response()->json([
                'success' => true,
                'message' => 'EMDBG Type Master deleted successfully.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete EMDBG Type Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Restore the specified soft deleted resource.
     */
    public function restore(string $id): JsonResponse
    {
        try {
            $emdbgType = EMDBGTypeMaster::withTrashed()->find($id);

            if (!$emdbgType) {
                return response()->json([
                    'success' => false,
                    'message' => 'EMDBG Type Master not found.'
                ], 404);
            }

            if (!$emdbgType->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'EMDBG Type Master is not deleted.'
                ], 400);
            }

            $emdbgType->restore();

            return response()->json([
                'success' => true,
                'message' => 'EMDBG Type Master restored successfully.',
                'data' => $this->transformEMDBGType($emdbgType)
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to restore EMDBG Type Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Permanently delete the specified resource.
     */
    public function forceDelete(string $id): JsonResponse
    {
        try {
            $emdbgType = EMDBGTypeMaster::withTrashed()->find($id);

            if (!$emdbgType) {
                return response()->json([
                    'success' => false,
                    'message' => 'EMDBG Type Master not found.'
                ], 404);
            }

            $emdbgType->forceDelete();

            return response()->json([
                'success' => true,
                'message' => 'EMDBG Type Master permanently deleted successfully.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to permanently delete EMDBG Type Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Bulk create EMDBG Type Masters.
     */
    public function bulkStore(Request $request): JsonResponse
    {
        // Validate request
        $validator = Validator::make($request->all(), [
            'types' => 'required|array|min:1',
            'types.*.emdbg_type_code' => 'nullable|string|max:20',
            'types.*.emdbg_type_name' => 'required|string|max:255',
            'types.*.percentage' => 'nullable|integer|min:0|max:100'
        ], [
            'types.required' => 'Types array is required.',
            'types.*.emdbg_type_name.required' => 'EMDBG Type Name is required for all items.',
            'types.*.emdbg_type_code.max' => 'EMDBG Type Code must not exceed 20 characters.',
            'types.*.emdbg_type_name.max' => 'EMDBG Type Name must not exceed 255 characters.',
            'types.*.percentage.integer' => 'Percentage must be an integer.',
            'types.*.percentage.min' => 'Percentage must be at least 0.',
            'types.*.percentage.max' => 'Percentage must not exceed 100.'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $createdTypes = [];
            $errors = [];

            foreach ($request->types as $index => $typeData) {
                // Check for duplicate type code if provided
                if (isset($typeData['emdbg_type_code']) && $typeData['emdbg_type_code']) {
                    $existing = EMDBGTypeMaster::where('emdbg_type_code', $typeData['emdbg_type_code'])->first();
                    if ($existing) {
                        $errors[] = [
                            'index' => $index,
                            'message' => 'EMDBG Type Code already exists: ' . $typeData['emdbg_type_code'],
                            'data' => $typeData
                        ];
                        continue;
                    }
                }

                // Check for duplicate type name
                $existingName = EMDBGTypeMaster::where('emdbg_type_name', $typeData['emdbg_type_name'])->first();
                if ($existingName) {
                    $errors[] = [
                        'index' => $index,
                        'message' => 'EMDBG Type Name already exists: ' . $typeData['emdbg_type_name'],
                        'data' => $typeData
                    ];
                    continue;
                }

                $emdbgType = EMDBGTypeMaster::create([
                    'emdbg_type_code' => $typeData['emdbg_type_code'] ?? null,
                    'emdbg_type_name' => $typeData['emdbg_type_name'],
                    'percentage' => $typeData['percentage'] ?? null
                ]);

                $createdTypes[] = $this->transformEMDBGType($emdbgType);
            }

            $response = [
                'success' => true,
                'message' => count($createdTypes) . ' EMDBG Type Master(s) created successfully.',
                'data' => $createdTypes
            ];

            if (!empty($errors)) {
                $response['errors'] = $errors;
                $response['message'] .= ' ' . count($errors) . ' item(s) failed.';
            }

            $statusCode = !empty($createdTypes) ? 201 : 400;

            return response()->json($response, $statusCode);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create EMDBG Type Masters.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get only active (non-deleted) EMDBG Type Masters.
     */
    public function active(): JsonResponse
    {
        try {
            $activeTypes = EMDBGTypeMaster::orderBy('emdbg_type_name', 'asc')->get();

            $transformedData = $activeTypes->map(function ($item) {
                return $this->transformEMDBGType($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Active EMDBG Type Masters retrieved successfully.',
                'data' => $transformedData
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve Active EMDBG Type Masters.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Search EMDBG Type Masters by various criteria.
     */
    public function search(Request $request): JsonResponse
    {
        try {
            $query = EMDBGTypeMaster::query();

            // Apply filters
            if ($request->has('emdbg_type_code')) {
                $query->where('emdbg_type_code', 'like', '%' . $request->emdbg_type_code . '%');
            }

            if ($request->has('emdbg_type_name')) {
                $query->where('emdbg_type_name', 'like', '%' . $request->emdbg_type_name . '%');
            }

            if ($request->has('percentage')) {
                $query->where('percentage', $request->percentage);
            }

            if ($request->has('percentage_range')) {
                $range = explode(',', $request->percentage_range);
                if (count($range) === 2) {
                    $query->whereBetween('percentage', [(int)$range[0], (int)$range[1]]);
                }
            }

            // Apply ordering
            $orderBy = $request->get('order_by', 'emdbg_id');
            $orderDirection = $request->get('order_direction', 'desc');
            $query->orderBy($orderBy, $orderDirection);

            $results = $query->get();

            $transformedData = $results->map(function ($item) {
                return $this->transformEMDBGType($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Search completed successfully.',
                'data' => $transformedData,
                'meta' => [
                    'total' => $results->count(),
                    'filters_applied' => $request->all()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Search failed.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Transform EMDBG Type data for API response.
     */
    private function transformEMDBGType($emdbgType): array
    {
        return [
            'emdbg_id' => $emdbgType->emdbg_id,
            'emdbg_type_code' => $emdbgType->emdbg_type_code,
            'emdbg_type_name' => $emdbgType->emdbg_type_name,
            'percentage' => $emdbgType->percentage,
            'percentage_formatted' => $emdbgType->percentage ? $emdbgType->percentage . '%' : null,
            'created_at' => $emdbgType->created_at ? $emdbgType->created_at->format('Y-m-d H:i:s') : null,
            'updated_at' => $emdbgType->updated_at ? $emdbgType->updated_at->format('Y-m-d H:i:s') : null,
            'deleted_at' => $emdbgType->deleted_at ? $emdbgType->deleted_at->format('Y-m-d H:i:s') : null,
            'is_deleted' => !is_null($emdbgType->deleted_at),
            'is_active' => is_null($emdbgType->deleted_at)
        ];
    }
}
