<?php

namespace App\Http\Controllers;

use App\Models\EmployeeCategoryMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class EmployeeCategoryMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = EmployeeCategoryMaster::orderBy('category_id', 'desc')->get();

            return response()->json([
                'success' => true,
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch records',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'category_code' => 'nullable|string|max:20|unique:employee_category_master,category_code',
                'category_name' => 'required|string|max:100',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $category = EmployeeCategoryMaster::create([
                'category_code' => $request->category_code,
                'category_name' => $request->category_name,
                'is_active' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Employee Category created successfully',
                'data' => $category
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Create failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $category = EmployeeCategoryMaster::find($id);

            if (!$category) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $category
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE (sometimes + update)
    public function update(Request $request, $id)
    {
        try {
            $category = EmployeeCategoryMaster::find($id);

            if (!$category) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'category_code' => 'sometimes|required|string|max:20|unique:employee_category_master,category_code,' . $id . ',category_id',
                'category_name' => 'sometimes|required|string|max:100',
                'is_active' => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            // ✅ correct update usage
            $category->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Employee Category updated successfully',
                'data' => $category
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $category = EmployeeCategoryMaster::find($id);

            if (!$category) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $category->delete();

            return response()->json([
                'success' => true,
                'message' => 'Employee Category deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Delete failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
