<?php

namespace App\Http\Controllers;

use App\Models\EmployeeMaster;
use App\Models\Department;
use App\Models\DesignationMaster;
use App\Models\EmployeeDesignationMaster;
use App\Models\EmployeeGradeMaster;
use App\Models\EmployeeTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class EmployeeMasterController extends Controller
{
    /**
     * Format employee data for response with foreign key relations
     */
    private function formatEmployeeWithRelations($employee)
    {
        $formatted = [
            'emp_id' => $employee->emp_id,
            'emp_code' => $employee->emp_code,
            'first_name' => $employee->first_name,
            'last_name' => $employee->last_name,
            'full_name' => trim($employee->first_name . ' ' . $employee->last_name),
            'doj' => $employee->doj ? $employee->doj->format('Y-m-d') : null,
            'is_active' => (bool) $employee->is_active,
            'status_text' => $employee->is_active ? 'Active' : 'Inactive',
            'created_at' => $employee->created_at ? $employee->created_at->format('Y-m-d H:i:s') : null,
            'updated_at' => $employee->updated_at ? $employee->updated_at->format('Y-m-d H:i:s') : null,
            'deleted_at' => $employee->deleted_at ? $employee->deleted_at->format('Y-m-d H:i:s') : null,
        ];

        // Add department data with eager loading
        if ($employee->dept_id && $employee->department) {
            $formatted['department'] = [
                'dept_id' => $employee->department->Dept_Id,
                'dept_code' => $employee->department->Dept_Code,
                'dept_name' => $employee->department->Dept_Name,
                'dept_type' => $employee->department->Dept_Type,
            ];
        } else {
            $formatted['department'] = null;
        }

        // Add designation data with eager loading
        if ($employee->designation_id && $employee->designation) {
            $formatted['designation'] = [
                'designation_id' => $employee->designation->designation_id,
                'designation_code' => $employee->designation->designation_code,
                'designation_name' => $employee->designation->designation_name,
                'grade_id' => $employee->designation->grade_id,
                'is_active' => (bool) $employee->designation->is_active,
            ];
        } else {
            $formatted['designation'] = null;
        }

        // Add grade data with eager loading
        if ($employee->grade_id && $employee->grade) {
            $formatted['grade'] = [
                'grade_id' => $employee->grade->grade_id,
                'grade_code' => $employee->grade->grade_code,
                'grade_name' => $employee->grade->grade_name,
                'pay_band' => $employee->grade->pay_band,
                'is_active' => (bool) $employee->grade->is_active,
            ];
        } else {
            $formatted['grade'] = null;
        }

        // Add employee type data with eager loading
        if ($employee->emp_type_id && $employee->employeeType) {
            $formatted['employee_type'] = [
                'emp_type_id' => $employee->employeeType->emp_type_id,
                'type_code' => $employee->employeeType->type_code,
                'type_name' => $employee->employeeType->type_name,
                'is_active' => (bool) $employee->employeeType->is_active,
            ];
        } else {
            $formatted['employee_type'] = null;
        }

        // Also include the IDs for backward compatibility
        $formatted['dept_id'] = $employee->dept_id;
        $formatted['designation_id'] = $employee->designation_id;
        $formatted['grade_id'] = $employee->grade_id;
        $formatted['emp_type_id'] = $employee->emp_type_id;

        return $formatted;
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            // Validate request parameters
            $validator = Validator::make($request->all(), [
                'search' => 'nullable|string|max:100',
                'is_active' => 'nullable|boolean',
                'dept_id' => 'nullable|integer|min:0',
                'designation_id' => 'nullable|integer|min:0',
                'grade_id' => 'nullable|integer|min:0',
                'emp_type_id' => 'nullable|integer|min:0',
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after_or_equal:start_date',
                'with_trashed' => 'nullable|boolean',
                'only_trashed' => 'nullable|boolean',
                'sort_by' => 'nullable|string|in:emp_id,emp_code,first_name,last_name,doj,created_at,updated_at',
                'sort_dir' => 'nullable|string|in:asc,desc',
                'per_page' => 'nullable|integer|min:1|max:100',
                'include_relations' => 'nullable|boolean', // New parameter to include relations
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $validated = $validator->validated();
            $includeRelations = $validated['include_relations'] ?? true; // Default to true

            $query = EmployeeMaster::query();

            // Eager load relations if requested
            if ($includeRelations) {
                $query->with(['department', 'designation', 'grade', 'employeeType']);
            }

            // Search functionality
            if (!empty($validated['search'])) {
                $query->search($validated['search']);
            }

            // Filter by active status
            if (isset($validated['is_active'])) {
                $query->where('is_active', $validated['is_active']);
            }

            // Filter by department
            if (isset($validated['dept_id'])) {
                $query->where('dept_id', $validated['dept_id']);
            }

            // Filter by designation
            if (isset($validated['designation_id'])) {
                $query->where('designation_id', $validated['designation_id']);
            }

            // Filter by grade
            if (isset($validated['grade_id'])) {
                $query->where('grade_id', $validated['grade_id']);
            }

            // Filter by employee type
            if (isset($validated['emp_type_id'])) {
                $query->where('emp_type_id', $validated['emp_type_id']);
            }

            // Date range filter
            if (!empty($validated['start_date'])) {
                $endDate = $validated['end_date'] ?? null;
                $query->dateRange($validated['start_date'], $endDate);
            }

            // Sorting
            $sortField = $validated['sort_by'] ?? 'emp_id';
            $sortDirection = $validated['sort_dir'] ?? 'desc';
            $query->orderBy($sortField, $sortDirection);

            // Include soft deleted if requested
            if (!empty($validated['with_trashed'])) {
                $query->withTrashed();
            }

            // Only trashed if requested
            if (!empty($validated['only_trashed'])) {
                $query->onlyTrashed();
            }

            // Pagination
            $perPage = $validated['per_page'] ?? 10;
            $employees = $query->paginate($perPage);

            // Format response data with or without relations
            $formattedData = $employees->map(function ($employee) use ($includeRelations) {
                if ($includeRelations) {
                    return $this->formatEmployeeWithRelations($employee);
                } else {
                    return [
                        'emp_id' => $employee->emp_id,
                        'emp_code' => $employee->emp_code,
                        'first_name' => $employee->first_name,
                        'last_name' => $employee->last_name,
                        'full_name' => trim($employee->first_name . ' ' . $employee->last_name),
                        'dept_id' => $employee->dept_id,
                        'designation_id' => $employee->designation_id,
                        'grade_id' => $employee->grade_id,
                        'emp_type_id' => $employee->emp_type_id,
                        'doj' => $employee->doj ? $employee->doj->format('Y-m-d') : null,
                        'is_active' => (bool) $employee->is_active,
                        'status_text' => $employee->is_active ? 'Active' : 'Inactive',
                        'created_at' => $employee->created_at ? $employee->created_at->format('Y-m-d H:i:s') : null,
                        'updated_at' => $employee->updated_at ? $employee->updated_at->format('Y-m-d H:i:s') : null,
                        'deleted_at' => $employee->deleted_at ? $employee->deleted_at->format('Y-m-d H:i:s') : null,
                    ];
                }
            });

            return response()->json([
                'success' => true,
                'message' => 'Employees retrieved successfully',
                'data' => $formattedData,
                'meta' => [
                    'total' => $employees->total(),
                    'per_page' => $employees->perPage(),
                    'current_page' => $employees->currentPage(),
                    'last_page' => $employees->lastPage(),
                    'from' => $employees->firstItem(),
                    'to' => $employees->lastItem(),
                ],
                'links' => [
                    'first' => $employees->url(1),
                    'last' => $employees->url($employees->lastPage()),
                    'prev' => $employees->previousPageUrl(),
                    'next' => $employees->nextPageUrl(),
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching employees: ' . $e->getMessage(), [
                'exception' => $e,
                'request' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch employees',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        DB::beginTransaction();

        try {
            // Validate request
            $validator = Validator::make($request->all(), [
                'emp_code' => [
                    'nullable',
                    'string',
                    'max:20',
                    Rule::unique('employee_master', 'emp_code')
                ],
                'first_name' => 'required|string|max:50',
                'last_name' => 'nullable|string|max:50',
                'dept_id' => [
                    'nullable',
                    'integer',
                    'min:0',
                    function ($attribute, $value, $fail) {
                        if ($value && !Department::where('Dept_Id', $value)->exists()) {
                            $fail('The selected department does not exist.');
                        }
                    }
                ],
                'designation_id' => [
                    'nullable',
                    'integer',
                    'min:0',
                    function ($attribute, $value, $fail) {
                        if ($value && !DesignationMaster::where('designation_id', $value)->exists()) {
                            $fail('The selected designation does not exist.');
                        }
                    }
                ],
                'grade_id' => [
                    'nullable',
                    'integer',
                    'min:0',
                    function ($attribute, $value, $fail) {
                        if ($value && !EmployeeGradeMaster::where('grade_id', $value)->exists()) {
                            $fail('The selected grade does not exist.');
                        }
                    }
                ],
                'emp_type_id' => [
                    'nullable',
                    'integer',
                    'min:0',
                    function ($attribute, $value, $fail) {
                        if ($value && !EmployeeTypeMaster::where('emp_type_id', $value)->exists()) {
                            $fail('The selected employee type does not exist.');
                        }
                    }
                ],
                'doj' => [
                    'nullable',
                    'date',
                    'before_or_equal:' . now()->format('Y-m-d')
                ],
                'is_active' => 'nullable|boolean'
            ], [
                'emp_code.required' => 'Employee code is required',
                'emp_code.unique' => 'This employee code already exists',
                'first_name.required' => 'First name is required',
                'doj.date' => 'Date of joining must be a valid date',
                'doj.before_or_equal' => 'Date of joining cannot be in the future'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $validatedData = $validator->validated();

            // Parse boolean values
            if (isset($validatedData['is_active']) && is_string($validatedData['is_active'])) {
                $validatedData['is_active'] = $validatedData['is_active'] === 'true' || $validatedData['is_active'] === '1';
            }

            // Set default values for nullable fields
            $validatedData['dept_id'] = $validatedData['dept_id'] ?? null;
            $validatedData['designation_id'] = $validatedData['designation_id'] ?? null;
            $validatedData['grade_id'] = $validatedData['grade_id'] ?? null;
            $validatedData['emp_type_id'] = $validatedData['emp_type_id'] ?? null;
            $validatedData['is_active'] = $validatedData['is_active'] ?? true;

            $employee = EmployeeMaster::create($validatedData);

            // Eager load relations for response
            $employee->load(['department', 'designation', 'grade', 'employeeType']);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Employee created successfully',
                'data' => $this->formatEmployeeWithRelations($employee)
            ], Response::HTTP_CREATED);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating employee: ' . $e->getMessage(), [
                'exception' => $e,
                'data' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to create employee',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            // Validate ID
            if (!is_numeric($id) || $id <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid employee ID'
                ], Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            // Eager load relations
            $employee = EmployeeMaster::with(['department', 'designation', 'grade', 'employeeType'])
                ->find($id);

            if (!$employee) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee not found'
                ], Response::HTTP_NOT_FOUND);
            }

            return response()->json([
                'success' => true,
                'message' => 'Employee retrieved successfully',
                'data' => $this->formatEmployeeWithRelations($employee)
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching employee: ' . $e->getMessage(), [
                'exception' => $e,
                'employee_id' => $id
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch employee',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            // Validate ID
            if (!is_numeric($id) || $id <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid employee ID'
                ], Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $employee = EmployeeMaster::find($id);

            if (!$employee) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee not found'
                ], Response::HTTP_NOT_FOUND);
            }

            // Validate request
            $validator = Validator::make($request->all(), [
                'emp_code' => [
                    'sometimes',
                    'string',
                    'max:20',
                    Rule::unique('employee_master', 'emp_code')->ignore($employee->emp_id, 'emp_id')
                ],
                'first_name' => 'sometimes|string|max:50',
                'last_name' => 'nullable|string|max:50',
                'dept_id' => [
                    'nullable',
                    'integer',
                    'min:0',
                    function ($attribute, $value, $fail) {
                        if ($value && !Department::where('Dept_Id', $value)->exists()) {
                            $fail('The selected department does not exist.');
                        }
                    }
                ],
                'designation_id' => [
                    'nullable',
                    'integer',
                    'min:0',
                    function ($attribute, $value, $fail) {
                        if ($value && !DesignationMaster::where('designation_id', $value)->exists()) {
                            $fail('The selected designation does not exist.');
                        }
                    }
                ],
                'grade_id' => [
                    'nullable',
                    'integer',
                    'min:0',
                    function ($attribute, $value, $fail) {
                        if ($value && !EmployeeGradeMaster::where('grade_id', $value)->exists()) {
                            $fail('The selected grade does not exist.');
                        }
                    }
                ],
                'emp_type_id' => [
                    'nullable',
                    'integer',
                    'min:0',
                    function ($attribute, $value, $fail) {
                        if ($value && !EmployeeTypeMaster::where('emp_type_id', $value)->exists()) {
                            $fail('The selected employee type does not exist.');
                        }
                    }
                ],
                'doj' => [
                    'nullable',
                    'date',
                    'before_or_equal:' . now()->format('Y-m-d')
                ],
                'is_active' => 'sometimes|boolean'
            ], [
                'emp_code.unique' => 'This employee code already exists',
                'doj.date' => 'Date of joining must be a valid date',
                'doj.before_or_equal' => 'Date of joining cannot be in the future'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $validatedData = $validator->validated();

            // Parse boolean values
            if (isset($validatedData['is_active']) && is_string($validatedData['is_active'])) {
                $validatedData['is_active'] = $validatedData['is_active'] === 'true' || $validatedData['is_active'] === '1';
            }

            // Ensure nullable fields are properly set
            if (array_key_exists('dept_id', $validatedData)) {
                $validatedData['dept_id'] = $validatedData['dept_id'] ?: null;
            }
            if (array_key_exists('designation_id', $validatedData)) {
                $validatedData['designation_id'] = $validatedData['designation_id'] ?: null;
            }
            if (array_key_exists('grade_id', $validatedData)) {
                $validatedData['grade_id'] = $validatedData['grade_id'] ?: null;
            }
            if (array_key_exists('emp_type_id', $validatedData)) {
                $validatedData['emp_type_id'] = $validatedData['emp_type_id'] ?: null;
            }

            $employee->update($validatedData);

            // Refresh and eager load relations for response
            $employee->refresh();
            $employee->load(['department', 'designation', 'grade', 'employeeType']);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Employee updated successfully',
                'data' => $this->formatEmployeeWithRelations($employee)
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating employee: ' . $e->getMessage(), [
                'exception' => $e,
                'employee_id' => $id,
                'data' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update employee',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        DB::beginTransaction();

        try {
            // Validate ID
            if (!is_numeric($id) || $id <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid employee ID'
                ], Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $employee = EmployeeMaster::find($id);

            if (!$employee) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee not found'
                ], Response::HTTP_NOT_FOUND);
            }

            $employee->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Employee deleted successfully'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting employee: ' . $e->getMessage(), [
                'exception' => $e,
                'employee_id' => $id
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete employee',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    public function getAllEmployees(Request $request)
    {
        try {
            // Validate only essential parameters
            $validator = Validator::make($request->all(), [
                'is_active' => 'nullable|boolean',
                'sort_by' => 'nullable|string|in:emp_id,emp_code,first_name,last_name,doj,created_at',
                'sort_dir' => 'nullable|string|in:asc,desc',
                'include_relations' => 'nullable|boolean',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $validated = $validator->validated();
            $includeRelations = $validated['include_relations'] ?? true;

            $query = EmployeeMaster::query();

            // Eager load relations if requested
            if ($includeRelations) {
                $query->with(['department', 'designation', 'grade', 'employeeType']);
            }

            // Filter by active status if provided
            if (isset($validated['is_active'])) {
                $query->where('is_active', $validated['is_active']);
            }

            // Sorting
            $sortField = $validated['sort_by'] ?? 'emp_id';
            $sortDirection = $validated['sort_dir'] ?? 'asc';
            $query->orderBy($sortField, $sortDirection);

            // Get all employees
            $employees = $query->get();

            // Get total count
            $totalCount = $employees->count();

            // Format response data
            $formattedData = $employees->map(function ($employee) use ($includeRelations) {
                if ($includeRelations) {
                    return $this->formatEmployeeWithRelations($employee);
                } else {
                    return [
                        'emp_id' => $employee->emp_id,
                        'emp_code' => $employee->emp_code,
                        'first_name' => $employee->first_name,
                        'last_name' => $employee->last_name,
                        'full_name' => trim($employee->first_name . ' ' . $employee->last_name),
                        'dept_id' => $employee->dept_id,
                        'designation_id' => $employee->designation_id,
                        'grade_id' => $employee->grade_id,
                        'emp_type_id' => $employee->emp_type_id,
                        'doj' => $employee->doj ? $employee->doj->format('Y-m-d') : null,
                        'is_active' => (bool) $employee->is_active,
                        'status_text' => $employee->is_active ? 'Active' : 'Inactive',
                        'created_at' => $employee->created_at ? $employee->created_at->format('Y-m-d H:i:s') : null,
                        'updated_at' => $employee->updated_at ? $employee->updated_at->format('Y-m-d H:i:s') : null,
                        'deleted_at' => $employee->deleted_at ? $employee->deleted_at->format('Y-m-d H:i:s') : null,
                    ];
                }
            });

            return response()->json([
                'success' => true,
                'message' => 'All employees retrieved successfully',
                'data' => $formattedData,
                'meta' => [
                    'total_count' => $totalCount,
                    'sort_by' => $sortField,
                    'sort_dir' => $sortDirection,
                    'include_relations' => $includeRelations,
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching all employees: ' . $e->getMessage(), [
                'exception' => $e,
                'request' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch employees',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
}
