<?php

namespace App\Http\Controllers;

use App\Models\EmployeeStatusMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class EmployeeStatusMasterController extends Controller
{
    // 🔹 INDEX
    public function index()
    {
        try {
            $data = EmployeeStatusMaster::orderBy('status_id', 'desc')->get();
            return response()->json([
                'success' => true,
                'message' => 'Employee statuses fetched successfully',
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch employee statuses',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'status_code' => 'nullable|string|max:20|unique:employee_status_masters,status_code',
                'status_name' => 'required|string|max:100',
                'is_active'   => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors'  => $validator->errors()
                ], 422);
            }

            $status = EmployeeStatusMaster::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Employee status created successfully',
                'data' => $status
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create employee status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $status = EmployeeStatusMaster::find($id);
            if (!$status) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee status not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $status
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch employee status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE (sometimes)
    public function update(Request $request, $id)
    {
        try {
            $status = EmployeeStatusMaster::find($id);
            if (!$status) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee status not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'status_code' => 'sometimes|required|string|max:20|unique:employee_status_masters,status_code,' . $id . ',status_id',
                'status_name' => 'sometimes|required|string|max:100',
                'is_active'   => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors'  => $validator->errors()
                ], 422);
            }

            $status->update($request->only(['status_code', 'status_name', 'is_active']));

            return response()->json([
                'success' => true,
                'message' => 'Employee status updated successfully',
                'data' => $status
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update employee status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $status = EmployeeStatusMaster::find($id);
            if (!$status) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee status not found'
                ], 404);
            }

            $status->delete();

            return response()->json([
                'success' => true,
                'message' => 'Employee status deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete employee status',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
