<?php

namespace App\Http\Controllers;

use App\Models\ExamIncrementMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class ExamIncrementMasterController extends Controller
{
    // 🔹 INDEX
    public function index()
    {
        try {
            $data = ExamIncrementMaster::orderBy('exam_type_id', 'desc')->get();
            return response()->json([
                'success' => true,
                'message' => 'Exam increments fetched successfully',
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch exam increments',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'exam_code' => 'nullable|string|max:30|unique:exam_increment_masters,exam_code',
                'exam_name' => 'required|string|max:100',
                'increment_amount' => 'nullable|numeric',
                'increment_details' => 'nullable|string|max:255',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $exam = ExamIncrementMaster::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Exam increment created successfully',
                'data' => $exam
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create exam increment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $exam = ExamIncrementMaster::find($id);
            if (!$exam) {
                return response()->json([
                    'success' => false,
                    'message' => 'Exam increment not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $exam
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch exam increment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE
    public function update(Request $request, $id)
    {
        try {
            $exam = ExamIncrementMaster::find($id);
            if (!$exam) {
                return response()->json([
                    'success' => false,
                    'message' => 'Exam increment not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'exam_code' => 'sometimes|required|string|max:30|unique:exam_increment_masters,exam_code,' . $id . ',exam_type_id',
                'exam_name' => 'sometimes|required|string|max:100',
                'increment_amount' => 'sometimes|numeric',
                'increment_details' => 'sometimes|string|max:255',
                'is_active' => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $exam->update($request->only([
                'exam_code',
                'exam_name',
                'increment_amount',
                'increment_details',
                'is_active'
            ]));

            return response()->json([
                'success' => true,
                'message' => 'Exam increment updated successfully',
                'data' => $exam
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update exam increment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $exam = ExamIncrementMaster::find($id);
            if (!$exam) {
                return response()->json([
                    'success' => false,
                    'message' => 'Exam increment not found'
                ], 404);
            }

            $exam->delete();

            return response()->json([
                'success' => true,
                'message' => 'Exam increment deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete exam increment',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
