<?php

namespace App\Http\Controllers;

use App\Models\FaTaxTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class FaTaxTypeMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = FaTaxTypeMaster::query();

        // Filter by ledger if provided
        if ($request->has('lg_id')) {
            $query->where('Lg_Id', $request->lg_id);
        }

        // Filter by tax code if provided
        if ($request->has('tax_code')) {
            $query->where('tax_code', 'like', '%' . $request->tax_code . '%');
        }

        // Filter by tax name if provided
        if ($request->has('tax_name')) {
            $query->where('tax_name', 'like', '%' . $request->tax_name . '%');
        }

        $taxTypes = $query->with('ledger')->get();

        return response()->json([
            'success' => true,
            'data' => $taxTypes
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate(FaTaxTypeMaster::createRules(), FaTaxTypeMaster::validationMessages());

        $taxType = FaTaxTypeMaster::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Tax type created successfully',
            'data' => $taxType->load('ledger')
        ], Response::HTTP_CREATED);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $taxType = FaTaxTypeMaster::with('ledger')->find($id);

        if (!$taxType) {
            return response()->json([
                'success' => false,
                'message' => 'Tax type not found'
            ], Response::HTTP_NOT_FOUND);
        }

        return response()->json([
            'success' => true,
            'data' => $taxType
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $taxType = FaTaxTypeMaster::find($id);

        if (!$taxType) {
            return response()->json([
                'success' => false,
                'message' => 'Tax type not found'
            ], Response::HTTP_NOT_FOUND);
        }

        $validated = $request->validate(FaTaxTypeMaster::updateRules($id), FaTaxTypeMaster::validationMessages());

        $taxType->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Tax type updated successfully',
            'data' => $taxType->load('ledger')
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $taxType = FaTaxTypeMaster::find($id);

        if (!$taxType) {
            return response()->json([
                'success' => false,
                'message' => 'Tax type not found'
            ], Response::HTTP_NOT_FOUND);
        }

        $taxType->delete();

        return response()->json([
            'success' => true,
            'message' => 'Tax type deleted successfully'
        ]);
    }

    /**
     * Get tax types by ledger ID
     */
    public function getByLedger($ledgerId)
    {
        $taxTypes = FaTaxTypeMaster::where('Lg_Id', $ledgerId)
            ->with('ledger')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $taxTypes
        ]);
    }

    /**
     * Get tax types without ledger association
     */
    public function getWithoutLedger()
    {
        $taxTypes = FaTaxTypeMaster::whereNull('Lg_Id')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $taxTypes
        ]);
    }

    /**
     * Bulk update ledger for multiple tax types
     */
    public function bulkUpdateLedger(Request $request)
    {
        $validated = $request->validate([
            'tax_type_ids' => 'required|array',
            'tax_type_ids.*' => 'exists:fa_tax_type_master,tax_type_id',
            'Lg_Id' => 'nullable|integer|exists:ledger_master,Lg_Id'
        ]);

        FaTaxTypeMaster::whereIn('tax_type_id', $validated['tax_type_ids'])
            ->update(['Lg_Id' => $validated['Lg_Id']]);

        return response()->json([
            'success' => true,
            'message' => 'Ledger updated for selected tax types'
        ]);
    }
}
