<?php

namespace App\Http\Controllers;

use App\Models\FamilyInfoMaster;
use App\Models\EmployeeMaster;
use App\Models\RelationshipMaster;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Throwable;
use Illuminate\Support\Facades\DB;

class FamilyInfoMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            // Get search query if any
            $query = $request->input('query');
            $isActive = $request->input('is_active');
            $relationId = $request->input('relation_id');
            $empId = $request->input('emp_id'); // ✅ Added new filter
            $gender = $request->input('gender'); // ✅ Added new filter

            $familyMembers = FamilyInfoMaster::with(['relationship', 'employee']); // ✅ Added employee relation

            // Apply search filter if query exists
            if ($query) {
                $familyMembers->where(function ($q) use ($query) {
                    $q->where('member_name', 'like', '%' . $query . '%')
                      ->orWhere('mob_num', 'like', '%' . $query . '%');
                });
            }

            // Filter by is_active if provided
            if ($isActive !== null) {
                $familyMembers->where('is_active', filter_var($isActive, FILTER_VALIDATE_BOOLEAN));
            }

            // Filter by relation_id if provided
            if ($relationId) {
                $familyMembers->where('relation_id', $relationId);
            }

            // ✅ Filter by emp_id if provided
            if ($empId) {
                $familyMembers->where('emp_id', $empId);
            }

            // ✅ Filter by gender if provided
            if ($gender) {
                $familyMembers->where('gender', $gender);
            }

            // Order by latest
            $familyMembers->orderBy('family_id', 'desc');

            $data = $familyMembers->get();

            return response()->json([
                'status' => 'success',
                'message' => 'Family members fetched successfully',
                'data' => $data
            ]);
        } catch (Throwable $e) {
            Log::error('Family Info Fetch Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch family members',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    /**
 * Get employees with their family relationships summary and full family details
 */
public function getEmployeeRelationshipsSummary(Request $request)
{
    try {
        // Get filters from request
        $empId = $request->input('emp_id');
        $isActive = $request->input('is_active');
        $hasFamily = $request->input('has_family'); // true/false to filter employees with/without family
        $relationId = $request->input('relation_id'); // ✅ New filter
        $gender = $request->input('gender'); // ✅ New filter
        $includeFullDetails = $request->input('include_full_details', true); // ✅ Control full details inclusion

        // Start query
        $query = EmployeeMaster::query();

        // Apply emp_id filter if provided
        if ($empId) {
            $query->where('emp_id', $empId);
        }

        // Apply is_active filter if provided
        if ($isActive !== null) {
            $query->where('is_active', filter_var($isActive, FILTER_VALIDATE_BOOLEAN));
        }

        // Select employees and their family data with full details
        $employees = $query->with(['familyMembers' => function($q) use ($relationId, $gender, $includeFullDetails) {
            // Apply family member filters
            if ($relationId) {
                $q->where('relation_id', $relationId);
            }
            
            if ($gender) {
                $q->where('gender', $gender);
            }
            
            // Always load relationship - check what columns exist
            $q->with(['relationship' => function($relQuery) {
                // First check if relation_name_hi column exists
                try {
                    // Try to get column names dynamically or use only relation_name
                    $relQuery->select('relation_id', 'relation_name');
                } catch (\Exception $e) {
                    // Fallback to basic selection
                    $relQuery->select('relation_id', 'relation_name');
                }
            }]);
            
            // Order family members
            $q->orderBy('relation_id')->orderBy('member_name');
        }])
        ->select([
            'emp_id',
            'emp_code',
            'first_name',
            'last_name',
            'is_active as emp_is_active'
        ])
        ->get();

        // Process each employee to get relationship summary with full details
        $result = $employees->map(function ($employee) use ($hasFamily, $includeFullDetails) {
            // Get family members
            $familyMembers = $employee->familyMembers;

            // Skip employees based on has_family filter
            if ($hasFamily === 'true' && $familyMembers->isEmpty()) {
                return null;
            }
            if ($hasFamily === 'false' && $familyMembers->isNotEmpty()) {
                return null;
            }

            // Extract unique relationship_ids with count and collect full details
            $relationships = [];
            $relationshipCounts = [];
            $familyDetails = []; // ✅ Store full family details
            
            // Gender counts
            $genderCounts = [
                'male' => 0,
                'female' => 0,
                'other' => 0
            ];

            foreach ($familyMembers as $member) {
                $relationId = $member->relation_id;

                // Count relationships
                if (!isset($relationshipCounts[$relationId])) {
                    $relationshipCounts[$relationId] = 0;
                }
                $relationshipCounts[$relationId]++;

                // Store relationship details
                if ($member->relationship && !isset($relationships[$relationId])) {
                    $relationships[$relationId] = [
                        'relation_id' => $relationId,
                        'relation_name' => $member->relationship->relation_name ?? null,
                        'count' => 0
                    ];
                }
                
                // ✅ Store full family member details
                $familyDetails[] = [
                    'family_id' => $member->family_id,
                    'member_name' => $member->member_name,
                    'gender' => $member->gender,
                    'relation_id' => $member->relation_id,
                    'relation_name' => $member->relationship->relation_name ?? null,
                    'dob' => $member->dob ? $member->dob->format('Y-m-d') : null,
                    'mob_num' => $member->mob_num,
                    'is_active' => (bool)$member->is_active,
                    'created_at' => $member->created_at ? $member->created_at->format('Y-m-d H:i:s') : null,
                    'updated_at' => $member->updated_at ? $member->updated_at->format('Y-m-d H:i:s') : null
                ];
                
                // Count genders
                if ($member->gender && isset($genderCounts[$member->gender])) {
                    $genderCounts[$member->gender]++;
                } elseif ($member->gender) {
                    $genderCounts['other']++;
                }
            }

            // Update counts in relationships array
            foreach ($relationships as $key => &$rel) {
                $rel['count'] = $relationshipCounts[$key] ?? 0;
            }

            // Convert to indexed array
            $relationshipsArray = array_values($relationships);
            
            // Calculate age for each family member (if DOB exists)
            if ($includeFullDetails) {
                foreach ($familyDetails as &$detail) {
                    if ($detail['dob']) {
                        $dob = \Carbon\Carbon::parse($detail['dob']);
                        $detail['age'] = $dob->age;
                        $detail['dob_formatted'] = $dob->format('d-m-Y');
                    } else {
                        $detail['age'] = null;
                        $detail['dob_formatted'] = null;
                    }
                }
            }

            $resultData = [
                'emp_id' => $employee->emp_id,
                'emp_code' => $employee->emp_code,
                'full_name' => trim($employee->first_name . ' ' . $employee->last_name),
                'first_name' => $employee->first_name,
                'last_name' => $employee->last_name,
                'emp_is_active' => (bool)$employee->emp_is_active,
                'total_family_members' => $familyMembers->count(),
                'has_family' => $familyMembers->isNotEmpty(),
                'relationships' => $relationshipsArray,
                'relationship_ids' => array_keys($relationshipCounts),
                'relationship_counts' => $relationshipCounts,
                'gender_summary' => $genderCounts,
                'active_family_members' => $familyMembers->where('is_active', true)->count(),
                'inactive_family_members' => $familyMembers->where('is_active', false)->count()
            ];
            
            // ✅ Add full family details if requested
            if ($includeFullDetails) {
                $resultData['family_details'] = $familyDetails;
            }

            return $resultData;
        })->filter(function ($item) {
            return $item !== null; // Remove null items from filtered results
        })->values(); // Reset array keys

        // Calculate overall summary
        $totalFamilyMembers = $result->sum('total_family_members');
        $totalRelationships = $result->flatMap(function($emp) {
            return $emp['relationships'];
        })->unique('relation_id')->count();
        
        // Gender summary across all employees
        $overallGenderCounts = [
            'male' => 0,
            'female' => 0,
            'other' => 0
        ];
        
        foreach ($result as $emp) {
            $overallGenderCounts['male'] += $emp['gender_summary']['male'];
            $overallGenderCounts['female'] += $emp['gender_summary']['female'];
            $overallGenderCounts['other'] += $emp['gender_summary']['other'];
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Employee relationships summary with family details fetched successfully',
            'data' => $result,
            'summary' => [
                'total_employees' => $result->count(),
                'employees_with_family' => $result->where('has_family', true)->count(),
                'employees_without_family' => $result->where('has_family', false)->count(),
                'total_family_members' => $totalFamilyMembers,
                'total_unique_relationships' => $totalRelationships,
                'gender_distribution' => $overallGenderCounts,
                'active_family_members' => $result->sum('active_family_members'),
                'inactive_family_members' => $result->sum('inactive_family_members')
            ],
            'filters_applied' => [
                'emp_id' => $empId,
                'is_active' => $isActive,
                'has_family' => $hasFamily,
                'relation_id' => $relationId,
                'gender' => $gender,
                'include_full_details' => $includeFullDetails
            ]
        ]);

    } catch (Throwable $e) {
        Log::error('Employee Relationships Summary Error: ' . $e->getMessage());
        return response()->json([
            'status' => 'error',
            'message' => 'Unable to fetch employee relationships summary',
            'error' => $e->getMessage()
        ], 500);
    }
}
   public function store(Request $request)
{
    try {
        // Validate request data
        $validated = $request->validate([
            'emp_id' => 'nullable|exists:employee_master,emp_id',
            'member_name' => 'required|string|max:100',
            'gender' => 'nullable|in:male,female,other',
            'relation_id' => 'nullable|exists:relationship_master,relation_id',
            'dob' => 'nullable|date|date_format:Y-m-d',
            'mob_num' => 'nullable|string|max:15|regex:/^[0-9]{10,15}$/',
            'is_active' => 'nullable|boolean',
        ]);

        // ✅ DEBUG: Check if emp_id exists in employee_master
        if (isset($validated['emp_id'])) {
            $employeeExists = \App\Models\EmployeeMaster::find($validated['emp_id']);
            Log::info('Employee check:', [
                'emp_id' => $validated['emp_id'],
                'exists' => $employeeExists ? 'Yes' : 'No',
                'employee' => $employeeExists
            ]);
        }

        // Create family member
        $familyMember = FamilyInfoMaster::create($validated);

        // ✅ DEBUG: Check what was saved
        Log::info('Family member saved:', [
            'saved_emp_id' => $familyMember->emp_id,
            'type' => gettype($familyMember->emp_id)
        ]);

        // Load relationships
        $familyMember->load(['relationship', 'employee']);

        // ✅ DEBUG: Check relationships after load
        Log::info('After load relationships:', [
            'employee_loaded' => $familyMember->employee ? 'Yes' : 'No',
            'relationship_loaded' => $familyMember->relationship ? 'Yes' : 'No'
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Family member created successfully',
            'data' => $familyMember
        ], 201);
    } catch (ValidationException $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Validation failed',
            'errors' => $e->errors()
        ], 422);
    } catch (Throwable $e) {
        Log::error('Family Info Create Error: ' . $e->getMessage());
        return response()->json([
            'status' => 'error',
            'message' => 'Unable to create family member',
            'error' => $e->getMessage()
        ], 500);
    }
}

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            // Find family member by family_id with relationships
            $familyMember = FamilyInfoMaster::with(['relationship', 'employee'])->find($id); // ✅ Added employee relation

            if (!$familyMember) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Family member not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Family member fetched successfully',
                'data' => $familyMember
            ]);
        } catch (Throwable $e) {
            Log::error('Family Info Show Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch family member',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $familyMember = FamilyInfoMaster::find($id);

            if (!$familyMember) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Family member not found'
                ], 404);
            }

            // Validate request data
            $validated = $request->validate([
                'emp_id' => 'sometimes|nullable|exists:employee_master,emp_id', // ✅ Added validation
                'member_name' => 'sometimes|required|string|max:100',
                'gender' => 'sometimes|nullable|in:male,female,other', // ✅ Added validation
                'relation_id' => 'sometimes|nullable|exists:relationship_master,relation_id',
                'dob' => 'sometimes|nullable|date|date_format:Y-m-d',
                'mob_num' => 'sometimes|nullable|string|max:15|regex:/^[0-9]{10,15}$/',
                'is_active' => 'sometimes|nullable|boolean',
            ]);

            // Update family member
            $familyMember->update($validated);
            $familyMember->load(['relationship', 'employee']); // ✅ Load employee relation

            return response()->json([
                'status' => 'success',
                'message' => 'Family member updated successfully',
                'data' => $familyMember
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('Family Info Update Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to update family member',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            // Find family member by family_id
            $familyMember = FamilyInfoMaster::find($id);

            if (!$familyMember) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Family member not found'
                ], 404);
            }

            // Delete family member
            $familyMember->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'Family member deleted successfully',
                'data' => null
            ]);
        } catch (Throwable $e) {
            Log::error('Family Info Delete Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to delete family member',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle active status of family member
     */
    public function toggleStatus($id)
    {
        try {
            // Find family member by family_id
            $familyMember = FamilyInfoMaster::find($id);

            if (!$familyMember) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Family member not found'
                ], 404);
            }

            // Toggle is_active status
            $familyMember->update([
                'is_active' => !$familyMember->is_active
            ]);

            $status = $familyMember->is_active ? 'Active' : 'Inactive';

            return response()->json([
                'status' => 'success',
                'message' => 'Family member status updated successfully',
                'data' => [
                    'family_id' => $familyMember->family_id,
                    'is_active' => $familyMember->is_active,
                    'status' => $status
                ]
            ]);
        } catch (Throwable $e) {
            Log::error('Family Info Toggle Status Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to toggle family member status',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
