<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\FasDebtTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FasDebtTypeMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $debtTypes = FasDebtTypeMaster::where('is_active', true)->get();
        return response()->json([
            'success' => true,
            'data' => $debtTypes,
            'options' => [
                'dt_cate_options' => FasDebtTypeMaster::getDtCateOptions(),
                'dt_lia_type_options' => FasDebtTypeMaster::getDtLiaTypeOptions()
            ]
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'debt_code' => 'nullable|string|max:20|unique:fas_debt_type_master,debt_code',
            'debt_name' => 'required|string|max:100',
            'dt_cate' => 'nullable|in:Long Term,Short Term',
            'dt_lia_type' => 'nullable|in:Payable,Deposit,Loan',
           
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $debtType = FasDebtTypeMaster::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Debt type created successfully',
            'data' => $debtType
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $debtType = FasDebtTypeMaster::find($id);

        if (!$debtType) {
            return response()->json([
                'success' => false,
                'message' => 'Debt type not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $debtType,
            'options' => [
                'dt_cate_options' => FasDebtTypeMaster::getDtCateOptions(),
                'dt_lia_type_options' => FasDebtTypeMaster::getDtLiaTypeOptions()
            ]
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $debtType = FasDebtTypeMaster::find($id);

        if (!$debtType) {
            return response()->json([
                'success' => false,
                'message' => 'Debt type not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'debt_code' => 'nullable|string|max:20|unique:fas_debt_type_master,debt_code,' . $id . ',debt_type_id',
            'debt_name' => 'required|string|max:100',
            'dt_cate' => 'nullable|in:Long Term,Short Term',
            'dt_lia_type' => 'nullable|in:Payable,Deposit,Loan',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $debtType->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Debt type updated successfully',
            'data' => $debtType
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $debtType = FasDebtTypeMaster::find($id);

        if (!$debtType) {
            return response()->json([
                'success' => false,
                'message' => 'Debt type not found'
            ], 404);
        }

        // Soft delete by marking as inactive
        $debtType->update(['is_active' => false]);

        return response()->json([
            'success' => true,
            'message' => 'Debt type deactivated successfully'
        ]);
    }

    /**
     * Restore the specified resource.
     */
    public function restore($id)
    {
        $debtType = FasDebtTypeMaster::find($id);

        if (!$debtType) {
            return response()->json([
                'success' => false,
                'message' => 'Debt type not found'
            ], 404);
        }

        $debtType->update(['is_active' => true]);

        return response()->json([
            'success' => true,
            'message' => 'Debt type activated successfully',
            'data' => $debtType
        ]);
    }

    /**
     * Get options for select dropdowns
     */
    public function getOptions()
    {
        return response()->json([
            'success' => true,
            'data' => [
                'dt_cate_options' => FasDebtTypeMaster::getDtCateOptions(),
                'dt_lia_type_options' => FasDebtTypeMaster::getDtLiaTypeOptions()
            ]
        ]);
    }
}
