<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\FasLoanTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FasLoanTypeMasterController extends Controller
{
    public function index()
    {
        $loanTypes = FasLoanTypeMaster::where('is_active', true)->get();
        return response()->json([
            'success' => true,
            'data' => $loanTypes
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'loan_code' => 'nullable|string|max:20|unique:fas_loan_type_master,loan_code',
            'loan_name' => 'required|string|max:100',
            'max_amount' => 'nullable|numeric|min:0',
            'loan_mode' => 'nullable|string|in:IN,OUT',
            'interest_rate' => 'nullable|numeric|between:0,99.99',
            'tenure_years' => 'nullable|integer|min:1',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $loanType = FasLoanTypeMaster::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Loan type created successfully',
            'data' => $loanType
        ], 201);
    }

    public function show($id)
    {
        $loanType = FasLoanTypeMaster::find($id);

        if (!$loanType) {
            return response()->json([
                'success' => false,
                'message' => 'Loan type not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $loanType
        ]);
    }

    public function update(Request $request, $id)
    {
        $loanType = FasLoanTypeMaster::find($id);

        if (!$loanType) {
            return response()->json([
                'success' => false,
                'message' => 'Loan type not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'loan_code' => 'nullable|string|max:20|unique:fas_loan_type_master,loan_code,' . $id . ',loan_type_id',
            'loan_name' => 'required|string|max:100',
            'max_amount' => 'nullable|numeric|min:0',
            'loan_mode' => 'nullable|string|in:IN,OUT',
            'interest_rate' => 'nullable|numeric|between:0,99.99',
            'tenure_years' => 'nullable|integer|min:1',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $loanType->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Loan type updated successfully',
            'data' => $loanType
        ]);
    }

    public function destroy($id)
    {
        $loanType = FasLoanTypeMaster::find($id);

        if (!$loanType) {
            return response()->json([
                'success' => false,
                'message' => 'Loan type not found'
            ], 404);
        }

        $loanType->update(['is_active' => false]);

        return response()->json([
            'success' => true,
            'message' => 'Loan type deactivated successfully'
        ]);
    }

    public function restore($id)
    {
        $loanType = FasLoanTypeMaster::find($id);

        if (!$loanType) {
            return response()->json([
                'success' => false,
                'message' => 'Loan type not found'
            ], 404);
        }

        $loanType->update(['is_active' => true]);

        return response()->json([
            'success' => true,
            'message' => 'Loan type activated successfully',
            'data' => $loanType
        ]);
    }
}
