<?php

namespace App\Http\Controllers;

use App\Models\FinancialYearMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;

class FinancialYearMasterController extends Controller
{
    /**
     * Display a listing of financial years.
     */
    public function index(Request $request)
    {
        try {
            $query = FinancialYearMaster::query();

            // Apply filters if provided
            if ($request->has('status')) {
                $query->where('status', $request->status);
            }

            if ($request->has('is_current_year')) {
                $query->where('is_current_year', $request->boolean('is_current_year'));
            }

            if ($request->has('is_budget_open')) {
                $query->where('is_budget_open', $request->boolean('is_budget_open'));
            }

            if ($request->has('is_transaction_open')) {
                $query->where('is_transaction_open', $request->boolean('is_transaction_open'));
            }

            // Order by start_date descending (latest first)
            $financialYears = $query->orderBy('start_date', 'desc')->get();

            return response()->json([
                'success' => true,
                'data' => $financialYears,
                'message' => 'Financial years retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching financial years: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve financial years.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created financial year.
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'financial_year_code' => 'required|string|max:9|unique:financial_year_master',
                'financial_year_name' => 'nullable|string|max:20',
                'start_date' => 'required|date',
                'end_date' => 'required|date|after:start_date',
                'is_current_year' => 'boolean',
                'is_budget_open' => 'boolean',
                'is_transaction_open' => 'boolean',
                'previous_year_id' => 'nullable|exists:financial_year_master,financial_year_id',
                'next_year_id' => 'nullable|exists:financial_year_master,financial_year_id',
                'remarks' => 'nullable|string|max:255',
                'status' => 'in:ACTIVE,CLOSED,LOCKED',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();

            // If this is set as current year, unset current year from others
            if ($request->boolean('is_current_year')) {
                FinancialYearMaster::where('is_current_year', true)->update(['is_current_year' => false]);
            }

            $financialYear = FinancialYearMaster::create($data);

            return response()->json([
                'success' => true,
                'data' => $financialYear,
                'message' => 'Financial year created successfully.'
            ], 201);
        } catch (\Exception $e) {
            Log::error('Error creating financial year: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to create financial year.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified financial year.
     */
    public function show($id)
    {
        try {
            $financialYear = FinancialYearMaster::with(['previousYear', 'nextYear'])->find($id);

            if (!$financialYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year not found.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $financialYear,
                'message' => 'Financial year retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching financial year: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve financial year.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified financial year.
     */
    public function update(Request $request, $id)
    {
        try {
            $financialYear = FinancialYearMaster::find($id);

            if (!$financialYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'financial_year_code' => [
                    'sometimes',
                    'string',
                    'max:9',
                    Rule::unique('financial_year_master')->ignore($financialYear->financial_year_id, 'financial_year_id')
                ],
                'financial_year_name' => 'nullable|string|max:20',
                'start_date' => 'sometimes|date',
                'end_date' => 'sometimes|date|after:start_date',
                'is_current_year' => 'boolean',
                'is_budget_open' => 'boolean',
                'is_transaction_open' => 'boolean',
                'is_month_end_closed' => 'boolean',
                'is_year_end_closed' => 'boolean',
                'previous_year_id' => 'nullable|exists:financial_year_master,financial_year_id',
                'next_year_id' => 'nullable|exists:financial_year_master,financial_year_id',
                'remarks' => 'nullable|string|max:255',
                'status' => 'sometimes|in:ACTIVE,CLOSED,LOCKED',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();

            // If setting as current year, unset current year from others
            if ($request->has('is_current_year') && $request->boolean('is_current_year')) {
                FinancialYearMaster::where('is_current_year', true)
                    ->where('financial_year_id', '!=', $id)
                    ->update(['is_current_year' => false]);
            }

            $financialYear->update($data);

            return response()->json([
                'success' => true,
                'data' => $financialYear,
                'message' => 'Financial year updated successfully.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating financial year: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update financial year.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified financial year.
     */
    public function destroy($id)
    {
        try {
            $financialYear = FinancialYearMaster::find($id);

            if (!$financialYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year not found.'
                ], 404);
            }

            // Check if this year is referenced as previous/next year by other records
            $referencedAsPrevious = FinancialYearMaster::where('previous_year_id', $id)->exists();
            $referencedAsNext = FinancialYearMaster::where('next_year_id', $id)->exists();

            if ($referencedAsPrevious || $referencedAsNext) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete financial year as it is referenced by other records.'
                ], 400);
            }

            $financialYear->delete();

            return response()->json([
                'success' => true,
                'message' => 'Financial year deleted successfully.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error deleting financial year: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete financial year.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get current financial year.
     */
    public function getCurrent()
    {
        try {
            $currentYear = FinancialYearMaster::where('is_current_year', true)->first();

            if (!$currentYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'No current financial year set.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $currentYear,
                'message' => 'Current financial year retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching current financial year: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve current financial year.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle budget open status.
     */
    public function toggleBudget($id)
    {
        try {
            $financialYear = FinancialYearMaster::find($id);

            if (!$financialYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year not found.'
                ], 404);
            }

            $financialYear->update([
                'is_budget_open' => !$financialYear->is_budget_open
            ]);

            return response()->json([
                'success' => true,
                'data' => $financialYear,
                'message' => 'Budget status toggled successfully.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error toggling budget status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle budget status.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle transaction open status.
     */
    public function toggleTransaction($id)
    {
        try {
            $financialYear = FinancialYearMaster::find($id);

            if (!$financialYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year not found.'
                ], 404);
            }

            $financialYear->update([
                'is_transaction_open' => !$financialYear->is_transaction_open
            ]);

            return response()->json([
                'success' => true,
                'data' => $financialYear,
                'message' => 'Transaction status toggled successfully.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error toggling transaction status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle transaction status.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
