<?php

namespace App\Http\Controllers;

use App\Models\FunctionType;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Throwable;

class FunctionTypeController extends Controller
{
    // GET all
    public function index(Request $request)
    {
        try {
            $queryValue = $request->input('query');
            $isvisible = $request->input('isvisible');

            $functionTypes = FunctionType::with('category', 'parentFunctionType', 'department');

            if ($queryValue) {
                $functionTypes->where(function ($q) use ($queryValue) {
                    $q->where('Upgrp', 'like', $queryValue . '%');
                });
            }

            // Filter by isvisible status if provided
            if ($isvisible !== null) {
                $functionTypes->where('isvisible', $isvisible);
            }

            $data = $functionTypes->get();

            return response()->json([
                'status' => 'success',
                'message' => 'Function types fetched successfully',
                'data' => $data
            ]);
        } catch (\Throwable $e) {
            Log::error('Fetch Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch function types',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    // GET one
    public function show($id)
    {
        try {
            $data = FunctionType::with('category', 'department')->find($id);

            if (!$data) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Function type not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Function type fetched successfully',
                'data' => $data
            ]);
        } catch (Throwable $e) {
            Log::error('Show Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch function type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // CREATE
    public function store(Request $request)
    {
        try {
            // Default Grsh
            $grsh = $request->has('Grsh') ? $request->input('Grsh') : '1';
            $grsrn = $request->has('Grsrn')
                ? $request->input('Grsrn')
                : FunctionType::where('Grsh', $grsh)->count() + 1;

            // Format code
            $prefix = str_pad($grsrn, 2, '0', STR_PAD_LEFT);
            $codeFormat = $prefix . '000000';

            $data = [
                'title'       => $request->input('title'), // ✅ added
                'department_id' => $request->input('department_id'), // ✅ ADD

                'desc'        => $request->input('desc'),
                'category_id' => $request->input('category_id'),
                'func_id'     => $request->input('func_id'),
                'Grsh'        => $grsh,
                'Grsrn'       => (string) $grsrn,
                'isvisible'   => $request->input('isvisible', 1), // Default to active
                'Upgrp'       => $request->input('Upgrp', $codeFormat),
                'Grcod'       => $request->input('Grcod', $codeFormat),
                'Menu_Links'  => $request->input('Menu_Links'),
            ];

            $type = FunctionType::create($data);

            return response()->json([
                'status' => 'success',
                'message' => 'Function type created successfully',
                'data' => $type
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('Create Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to create function type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // UPDATE
    // UPDATE
    public function update(Request $request, $id)
    {
        try {
            $type = FunctionType::find($id);

            if (!$type) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Function type not found'
                ], 404);
            }

            $data = $request->validate([
                'title'       => 'sometimes|nullable|string', // ✅ added
                'department_id' => 'sometimes|nullable|exists:main_departments,id', // ✅ ADD

                'desc'        => 'sometimes|nullable|string',
                'category_id' => 'sometimes|nullable|exists:categories,category_id',
                'Upgrp'       => 'sometimes|nullable|string',
                'Grcod'       => 'sometimes|nullable|string',
                'Grsrn'       => 'sometimes|nullable|string',
                'Grsh'        => 'sometimes|nullable|string',
                'isvisible'   => 'sometimes|nullable|integer|in:0,1', // Added validation
                'func_id'     => 'sometimes|nullable|exists:function_types,id',
                'Menu_Links'  => 'sometimes|nullable|string',
            ]);

            $type->update($data);

            return response()->json([
                'status' => 'success',
                'message' => 'Function type updated successfully',
                'data' => $type
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('Update Error: ' . $e->getMessage()); // Fixed: Added $ before e
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to update function type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // TOGGLE VISIBILITY
    public function toggleVisibility($id)
    {
        try {
            $type = FunctionType::find($id);

            if (!$type) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Function type not found'
                ], 404);
            }

            // Toggle between 0 and 1
            $type->update([
                'isvisible' => $type->isvisible == 1 ? 0 : 1
            ]);

            $status = $type->isvisible == 1 ? 'Active' : 'Inactive';

            return response()->json([
                'status' => 'success',
                'message' => 'Function type visibility toggled successfully',
                'data' => [
                    'id' => $type->id,
                    'isvisible' => $type->isvisible,
                    'status' => $status
                ]
            ]);
        } catch (Throwable $e) {
            Log::error('Toggle Visibility Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to toggle function type visibility',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // DELETE
    public function destroy($id)
    {
        try {
            $type = FunctionType::find($id);

            if (!$type) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Function type not found'
                ], 404);
            }

            $type->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'Function type deleted successfully'
            ]);
        } catch (Throwable $e) {
            Log::error('Delete Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to delete function type',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
