<?php

namespace App\Http\Controllers;

use App\Models\FunctionTypesTwo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Throwable;

class FunctionTypesTwoController extends Controller
{
    // GET ALL
    public function index(Request $request)
    {
        try {
            $queryValue = $request->query('query');
            $isvisible  = $request->query('isvisible');

            $functions = FunctionTypesTwo::query();

            if ($queryValue) {
                $functions->where('Upgrp', 'like', $queryValue . '%');
            }

            if ($isvisible !== null) {
                $functions->where('isvisible', $isvisible);
            }

            return response()->json([
                'status' => 'success',
                'data'   => $functions->get()
            ]);
        } catch (Throwable $e) {
            Log::error($e->getMessage());
            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to fetch function types'
            ], 500);
        }
    }

    // GET ONE
    public function show($id)
    {
        $data = FunctionTypesTwo::find($id);

        if (!$data) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Function type not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data'   => $data
        ]);
    }

    // CREATE (SAME LOGIC AS DesignationMenu)
    public function store(Request $request)
    {
        try {
            $grsh  = $request->input('Grsh', '1');
            $grsrn = FunctionTypesTwo::where('Grsh', $grsh)->count() + 1;

            $prefix = str_pad($grsrn, 2, '0', STR_PAD_LEFT);
            $code   = $prefix . '000000';

            $data = FunctionTypesTwo::create([
                'title'      => $request->title,
                'desc'       => $request->desc,
                'Grsh'       => $grsh,
                'Grsrn'      => (string) $grsrn,
                'Upgrp'      => $request->Upgrp ?? $code,
                'Grcod'      => $request->Grcod ?? $code,
                'Menu_Links' => $request->Menu_Links,
                'isvisible'  => $request->isvisible ?? 1,
            ]);

            return response()->json([
                'status' => 'success',
                'data'   => $data
            ], 201);
        } catch (Throwable $e) {
            Log::error('FunctionTypesTwo Store Error: ' . $e->getMessage());

            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to create function type'
            ], 500);
        }
    }

    // UPDATE
    public function update(Request $request, $id)
    {
        try {
            $data = FunctionTypesTwo::find($id);

            if (!$data) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Function type not found'
                ], 404);
            }

            $validated = $request->validate([
                'title'      => 'sometimes|nullable|string',
                'desc'       => 'sometimes|nullable|string',
                'Menu_Links' => 'sometimes|nullable|string',
                'Upgrp'      => 'sometimes|nullable|string',
                'Grcod'      => 'sometimes|nullable|string',
                'Grsrn'      => 'sometimes|nullable|string',
                'Grsh'       => 'sometimes|nullable|string',
                'isvisible'  => 'sometimes|integer|in:0,1',
            ]);

            $data->update($validated);

            return response()->json([
                'status'  => 'success',
                'message' => 'Function type updated successfully',
                'data'    => $data
            ]);
        } catch (Throwable $e) {
            Log::error('FunctionTypesTwo Update Error: ' . $e->getMessage());

            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to update function type'
            ], 500);
        }
    }

    // TOGGLE VISIBILITY
    public function toggleVisibility($id)
    {
        try {
            $data = FunctionTypesTwo::find($id);

            if (!$data) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Function type not found'
                ], 404);
            }

            $data->isvisible = $data->isvisible == 1 ? 0 : 1;
            $data->save();

            return response()->json([
                'status'  => 'success',
                'message' => 'Visibility toggled successfully',
                'data'    => $data
            ]);
        } catch (Throwable $e) {
            Log::error('FunctionTypesTwo Toggle Error: ' . $e->getMessage());

            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to toggle visibility'
            ], 500);
        }
    }

    // DELETE
    public function destroy($id)
    {
        $data = FunctionTypesTwo::find($id);

        if (!$data) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Function type not found'
            ], 404);
        }

        $data->delete();

        return response()->json([
            'status'  => 'success',
            'message' => 'Function type deleted successfully'
        ]);
    }
}
