<?php

namespace App\Http\Controllers;

use App\Models\GmcDeviceRegistrationQueue;
use App\Models\GmcDeviceApproval;
use App\Models\GmcDeviceApprovalLog;
use App\Models\GmcUser;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class GmcDeviceRegistrationQueueController extends Controller
{
    /**
     * Display listing of device registration queue
     */
    // public function index(Request $request)
    // {
    //     try {
    //         // Log the request
    //         Log::info('Device Queue Index Request:', $request->all());

    //         $perPage = (int) $request->get('per_page', 20);
    //         $page = (int) $request->get('page', 1);

    //         // Start query
    //         $query = GmcDeviceRegistrationQueue::query();

    //         // Load relationships safely
    //         $query->with([
    //             'user' => function ($q) {
    //                 $q->select('user_id', 'username', 'email', 'mobile', 'Role_Id', 'is_active');
    //             },
    //             'processor' => function ($q) {
    //                 $q->select('user_id', 'username');
    //             }
    //         ]);

    //         // Apply filters
    //         if ($request->filled('status')) {
    //             $query->where('status', $request->get('status'));
    //         }

    //         if ($request->filled('is_citizen')) {
    //             $query->where('is_citizen', $request->boolean('is_citizen'));
    //         }

    //         if ($request->filled('search')) {
    //             $search = $request->get('search');
    //             $query->where(function ($q) use ($search) {
    //                 $q->where('ip_address', 'LIKE', "%{$search}%")
    //                     ->orWhere('device_name', 'LIKE', "%{$search}%")
    //                     ->orWhere('mac_address', 'LIKE', "%{$search}%")
    //                     ->orWhereHas('user', function ($userQuery) use ($search) {
    //                         $userQuery->where('username', 'LIKE', "%{$search}%")
    //                             ->orWhere('email', 'LIKE', "%{$search}%");
    //                     });
    //             });
    //         }

    //         // Order and paginate
    //         $query->orderByDesc('queue_id');
    //         $queues = $query->paginate($perPage, ['*'], 'page', $page);

    //         // Transform data
    //         $transformedData = [];
    //         foreach ($queues as $queue) {
    //             $transformedData[] = [
    //                 'queue_id' => $queue->queue_id,
    //                 'user_id' => $queue->user_id,
    //                 'username' => $queue->user ? $queue->user->username : 'N/A',
    //                 'email' => $queue->user ? $queue->user->email : 'N/A',
    //                 'mobile' => $queue->user ? $queue->user->mobile : 'N/A',
    //                 'ip_address' => $queue->ip_address,
    //                 'device_name' => $queue->device_name,
    //                 'device_type' => $queue->device_type,
    //                 'operating_system' => $queue->operating_system,
    //                 'browser' => $queue->browser,
    //                 'mac_address' => $queue->mac_address,
    //                 'browser_fingerprint' => $queue->browser_fingerprint,
    //                 'status' => $queue->status,
    //                 'is_citizen' => (bool) $queue->is_citizen,
    //                 'submitted_at' => $queue->submitted_at ? $queue->submitted_at->format('Y-m-d H:i:s') : null,
    //                 'processed_at' => $queue->processed_at ? $queue->processed_at->format('Y-m-d H:i:s') : null,
    //                 'processed_by' => $queue->processor ? $queue->processor->username : null,
    //                 'rejection_reason' => $queue->rejection_reason
    //             ];
    //         }

    //         // Get counts
    //         $pendingCount = GmcDeviceRegistrationQueue::where('status', 'PENDING')->count();
    //         $approvedCount = GmcDeviceRegistrationQueue::where('status', 'APPROVED')->count();
    //         $rejectedCount = GmcDeviceRegistrationQueue::where('status', 'REJECTED')->count();

    //         return response()->json([
    //             'success' => true,
    //             'message' => 'Device registration queue fetched successfully',
    //             'data' => $transformedData,
    //             'meta' => [
    //                 'current_page' => $queues->currentPage(),
    //                 'per_page' => $queues->perPage(),
    //                 'total' => $queues->total(),
    //                 'last_page' => $queues->lastPage(),
    //                 'from' => $queues->firstItem(),
    //                 'to' => $queues->lastItem(),
    //                 'has_more_pages' => $queues->hasMorePages(),
    //                 'pending_count' => $pendingCount,
    //                 'approved_count' => $approvedCount,
    //                 'rejected_count' => $rejectedCount
    //             ]
    //         ], 200);
    //     } catch (\Exception $e) {
    //         Log::error('Device Queue Index Error: ' . $e->getMessage());
    //         Log::error('Stack Trace: ' . $e->getTraceAsString());

    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Unable to fetch device registration queue: ' .
    //                 (env('APP_DEBUG') ? $e->getMessage() : 'Internal server error')
    //         ], 500);
    //     }
    // }
    /**
     * Display listing of device registration queue
     */
    public function index(Request $request)
    {
        try {
            // Log the request
            Log::info('Device Queue Index Request:', $request->all());

            $perPage = (int) $request->get('per_page', 20);
            $page = (int) $request->get('page', 1);

            // Start query
            $query = GmcDeviceRegistrationQueue::query();

            // Load relationships safely
            $query->with([
                'user' => function ($q) {
                    $q->select('user_id', 'username', 'email', 'mobile', 'Role_Id', 'is_active');
                },
                'processor' => function ($q) {
                    $q->select('user_id', 'username');
                },
                'department' => function ($q) {
                    $q->select('department_id', 'department_name');
                }
            ]);

            // Apply filters
            if ($request->filled('status')) {
                $query->where('status', $request->get('status'));
            }

            if ($request->filled('is_citizen')) {
                $query->where('is_citizen', $request->boolean('is_citizen'));
            }

            if ($request->filled('department_id')) {
                $query->where('department_id', $request->get('department_id'));
            }

            if ($request->filled('device_type')) {
                $query->where('device_type', $request->get('device_type'));
            }

            if ($request->filled('search')) {
                $search = $request->get('search');
                $query->where(function ($q) use ($search) {
                    $q->where('queue_id', 'LIKE', "%{$search}%")
                        ->orWhere('ip_address', 'LIKE', "%{$search}%")
                        ->orWhere('device_name', 'LIKE', "%{$search}%")
                        ->orWhere('mac_address', 'LIKE', "%{$search}%")
                        ->orWhere('hdd_serial', 'LIKE', "%{$search}%")
                        ->orWhere('motherboard_serial', 'LIKE', "%{$search}%")
                        ->orWhere('processor_id', 'LIKE', "%{$search}%")
                        ->orWhere('browser_fingerprint', 'LIKE', "%{$search}%")
                        ->orWhere('operating_system', 'LIKE', "%{$search}%")
                        ->orWhere('browser', 'LIKE', "%{$search}%")
                        ->orWhereHas('user', function ($userQuery) use ($search) {
                            $userQuery->where('username', 'LIKE', "%{$search}%")
                                ->orWhere('email', 'LIKE', "%{$search}%")
                                ->orWhere('mobile', 'LIKE', "%{$search}%");
                        });
                });
            }

            // Order and paginate
            $query->orderByDesc('queue_id');
            $queues = $query->paginate($perPage, ['*'], 'page', $page);

            // Transform data - Include ALL fields from the table
            $transformedData = [];
            foreach ($queues as $queue) {
                $transformedData[] = [
                    // Primary ID
                    'queue_id' => $queue->queue_id,

                    // User Information
                    'user_id' => $queue->user_id,
                    'user_info' => $queue->user ? [
                        'username' => $queue->user->username,
                        'email' => $queue->user->email,
                        'mobile' => $queue->user->mobile,
                        'role_id' => $queue->user->Role_Id,
                        'is_active' => (bool) $queue->user->is_active
                    ] : null,

                    // Registration Data
                    'registration_data' => $queue->registration_data ?? [],

                    // Network Information
                    'ip_address' => $queue->ip_address,
                    'mac_address' => $queue->mac_address,

                    // Hardware Information
                    'hdd_serial' => $queue->hdd_serial,
                    'motherboard_serial' => $queue->motherboard_serial,
                    'processor_id' => $queue->processor_id,

                    // Browser & Software Information
                    'browser_fingerprint' => $queue->browser_fingerprint,
                    'operating_system' => $queue->operating_system,
                    'browser' => $queue->browser,

                    // Device Information
                    'device_name' => $queue->device_name,
                    'device_type' => $queue->device_type,

                    // Organizational Information
                    'department_id' => $queue->department_id,
                    'department' => $queue->department ? [
                        'department_id' => $queue->department->department_id,
                        'department_name' => $queue->department->department_name
                    ] : null,

                    // Status Information
                    'status' => $queue->status,
                    'is_citizen' => (bool) $queue->is_citizen,
                    'rejection_reason' => $queue->rejection_reason,

                    // Timestamps
                    'submitted_at' => $queue->submitted_at ? $queue->submitted_at->format('Y-m-d H:i:s') : null,
                    'processed_at' => $queue->processed_at ? $queue->processed_at->format('Y-m-d H:i:s') : null,

                    // Processor Information
                    'processed_by' => $queue->processed_by,
                    'processor_info' => $queue->processor ? [
                        'user_id' => $queue->processor->user_id,
                        'username' => $queue->processor->username
                    ] : null,

                    // Additional computed fields
                    'device_unique_id' => $queue->generateDeviceUniqueId(),
                    'is_pending' => $queue->isPending(),
                    'is_approved' => $queue->isApproved(),
                    'is_rejected' => $queue->isRejected(),
                ];
            }

            // Get counts
            $pendingCount = GmcDeviceRegistrationQueue::where('status', 'PENDING')->count();
            $approvedCount = GmcDeviceRegistrationQueue::where('status', 'APPROVED')->count();
            $rejectedCount = GmcDeviceRegistrationQueue::where('status', 'REJECTED')->count();
            $expiredCount = GmcDeviceRegistrationQueue::where('status', 'EXPIRED')->count();

            return response()->json([
                'success' => true,
                'message' => 'Device registration queue fetched successfully',
                'data' => $transformedData,
                'meta' => [
                    'current_page' => $queues->currentPage(),
                    'per_page' => $queues->perPage(),
                    'total' => $queues->total(),
                    'last_page' => $queues->lastPage(),
                    'from' => $queues->firstItem(),
                    'to' => $queues->lastItem(),
                    'has_more_pages' => $queues->hasMorePages(),
                    'pending_count' => $pendingCount,
                    'approved_count' => $approvedCount,
                    'rejected_count' => $rejectedCount,
                    'expired_count' => $expiredCount,
                    'citizen_count' => GmcDeviceRegistrationQueue::where('is_citizen', true)->count(),
                    'employee_count' => GmcDeviceRegistrationQueue::where('is_citizen', false)->count()
                ]
            ], 200);
        } catch (\Exception $e) {
            Log::error('Device Queue Index Error: ' . $e->getMessage());
            Log::error('Stack Trace: ' . $e->getTraceAsString());

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch device registration queue: ' .
                    (env('APP_DEBUG') ? $e->getMessage() : 'Internal server error')
            ], 500);
        }
    }


    
    /**
     * Show single device registration queue entry
     */
    public function show($id)
    {
        try {
            $queue = GmcDeviceRegistrationQueue::with([
                'user' => function ($q) {
                    $q->select('user_id', 'username', 'email', 'mobile', 'Role_Id', 'is_active', 'created_at');
                },
                'processor' => function ($q) {
                    $q->select('user_id', 'username', 'email');
                }
            ])->find($id);

            if (!$queue) {
                return response()->json([
                    'success' => false,
                    'message' => 'Device registration not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Device registration details fetched successfully',
                'data' => [
                    'queue_id' => $queue->queue_id,
                    'user' => $queue->user ? [
                        'user_id' => $queue->user->user_id,
                        'username' => $queue->user->username,
                        'email' => $queue->user->email,
                        'mobile' => $queue->user->mobile,
                        'Role_Id' => $queue->user->Role_Id,
                        'is_active' => $queue->user->is_active
                    ] : null,
                    'registration_data' => $queue->registration_data ?? [],
                    'device_info' => [
                        'ip_address' => $queue->ip_address,
                        'mac_address' => $queue->mac_address,
                        'device_name' => $queue->device_name,
                        'device_type' => $queue->device_type,
                        'operating_system' => $queue->operating_system,
                        'browser' => $queue->browser,
                        'browser_fingerprint' => $queue->browser_fingerprint,
                        'hdd_serial' => $queue->hdd_serial,
                        'motherboard_serial' => $queue->motherboard_serial,
                        'processor_id' => $queue->processor_id
                    ],
                    'status_info' => [
                        'status' => $queue->status,
                        'is_citizen' => (bool) $queue->is_citizen,
                        'submitted_at' => $queue->submitted_at ? $queue->submitted_at->format('Y-m-d H:i:s') : null,
                        'processed_at' => $queue->processed_at ? $queue->processed_at->format('Y-m-d H:i:s') : null,
                        'processed_by' => $queue->processor ? [
                            'user_id' => $queue->processor->user_id,
                            'username' => $queue->processor->username,
                            'email' => $queue->processor->email
                        ] : null,
                        'rejection_reason' => $queue->rejection_reason
                    ],
                    'device_unique_id' => $queue->generateDeviceUniqueId()
                ]
            ], 200);
        } catch (\Exception $e) {
            Log::error('Device Queue Show Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch device registration details'
            ], 500);
        }
    }

    /**
     * Approve device registration
     */
    public function approve(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'remarks' => 'nullable|string|max:500',
                'validity_days' => 'integer|min:1|max:3650'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $queue = GmcDeviceRegistrationQueue::with('user')->find($id);

            if (!$queue) {
                return response()->json([
                    'success' => false,
                    'message' => 'Device registration not found'
                ], 404);
            }

            if ($queue->status === 'APPROVED') {
                return response()->json([
                    'success' => false,
                    'message' => 'Device registration already approved'
                ], 400);
            }

            // Generate device unique ID
            $deviceUniqueId = $queue->generateDeviceUniqueId();

            // Set validity period
            $validityDays = $request->get('validity_days', $queue->is_citizen ? 3650 : 365);
            $validUntil = now()->addDays($validityDays);

            // Create device approval
            $approvalData = [
                'user_id' => $queue->user_id,
                'device_unique_id' => $deviceUniqueId,
                'device_name' => $queue->device_name,
                'device_type' => $queue->device_type,
                'device_category' => $queue->is_citizen ? 'CITIZEN' : 'EMPLOYEE',
                'ip_address' => $queue->ip_address,
                'mac_address' => $queue->mac_address,
                'hdd_serial' => $queue->hdd_serial,
                'motherboard_serial' => $queue->motherboard_serial,
                'processor_id' => $queue->processor_id,
                'operating_system' => $queue->operating_system,
                'browser' => $queue->browser,
                'browser_fingerprint' => $queue->browser_fingerprint,
                'approved_by' => auth()->id(),
                'approved_at' => now(),
                'valid_from' => now(),
                'valid_until' => $validUntil,
                'status' => 'APPROVED',
                'is_active' => true,
                'is_primary_device' => true,
                'remarks' => $request->get('remarks', 'Approved by administrator')
            ];

            // Add optional fields if they exist
            if ($queue->department_id) {
                $approvalData['department_id'] = $queue->department_id;
            }
            if ($queue->location_id) {
                $approvalData['location_id'] = $queue->location_id;
            }

            $approval = GmcDeviceApproval::create($approvalData);

            // Update queue status
            $queue->update([
                'status' => 'APPROVED',
                'processed_at' => now(),
                'processed_by' => auth()->id()
            ]);

            // Log approval
            if (class_exists('App\Models\GmcDeviceApprovalLog')) {
                GmcDeviceApprovalLog::logApproval(
                    $queue->queue_id,
                    $approval->approval_id,
                    $queue->user_id,
                    auth()->id(),
                    'MANUAL_APPROVAL',
                    $request->get('remarks', 'Device approved by administrator')
                );
            }

            return response()->json([
                'success' => true,
                'message' => 'Device registration approved successfully',
                'data' => [
                    'queue_id' => $queue->queue_id,
                    'approval_id' => $approval->approval_id,
                    'device_unique_id' => $deviceUniqueId,
                    'valid_until' => $validUntil->format('Y-m-d H:i:s'),
                    'validity_days' => $validityDays
                ]
            ], 200);
        } catch (\Exception $e) {
            Log::error('Device Approval Error: ' . $e->getMessage());
            Log::error('Stack Trace: ' . $e->getTraceAsString());

            return response()->json([
                'success' => false,
                'message' => 'Failed to approve device registration: ' .
                    (env('APP_DEBUG') ? $e->getMessage() : 'Internal server error')
            ], 500);
        }
    }

    /**
     * Reject device registration
     */
    public function reject(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'rejection_reason' => 'required|string|max:1000'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $queue = GmcDeviceRegistrationQueue::find($id);

            if (!$queue) {
                return response()->json([
                    'success' => false,
                    'message' => 'Device registration not found'
                ], 404);
            }

            if ($queue->status === 'REJECTED') {
                return response()->json([
                    'success' => false,
                    'message' => 'Device registration already rejected'
                ], 400);
            }

            $queue->update([
                'status' => 'REJECTED',
                'rejection_reason' => $request->rejection_reason,
                'processed_at' => now(),
                'processed_by' => auth()->id()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Device registration rejected successfully'
            ], 200);
        } catch (\Exception $e) {
            Log::error('Device Rejection Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to reject device registration'
            ], 500);
        }
    }

    /**
     * Get statistics
     */
    public function statistics()
    {
        try {
            $total = GmcDeviceRegistrationQueue::count();
            $pending = GmcDeviceRegistrationQueue::where('status', 'PENDING')->count();
            $approved = GmcDeviceRegistrationQueue::where('status', 'APPROVED')->count();
            $rejected = GmcDeviceRegistrationQueue::where('status', 'REJECTED')->count();

            return response()->json([
                'success' => true,
                'message' => 'Device registration statistics',
                'data' => [
                    'total' => $total,
                    'pending' => $pending,
                    'approved' => $approved,
                    'rejected' => $rejected,
                    'approval_rate' => $total > 0 ? round(($approved / $total) * 100, 2) : 0
                ]
            ], 200);
        } catch (\Exception $e) {
            Log::error('Device Statistics Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch statistics'
            ], 500);
        }
    }

    /**
     * Get pending count
     */
    public function pendingCount()
    {
        try {
            $count = GmcDeviceRegistrationQueue::where('status', 'PENDING')->count();
            return response()->json([
                'success' => true,
                'message' => 'Pending registrations count',
                'data' => ['pending_count' => $count]
            ], 200);
        } catch (\Exception $e) {
            Log::error('Pending Count Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch pending count'
            ], 500);
        }
    }

    /**
     * Test endpoint to check if controller is working
     */
    public function test()
    {
        try {
            $count = GmcDeviceRegistrationQueue::count();

            return response()->json([
                'success' => true,
                'message' => 'Device Registration Queue Controller is working',
                'data' => [
                    'total_records' => $count,
                    'controller' => 'GmcDeviceRegistrationQueueController',
                    'status' => 'OK'
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Controller test failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
