<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\GmcUser;
use App\Models\RoleMaster;
use App\Models\GmcUserAuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Hash;

class GmcUserController extends Controller
{
    /**
     * Display a listing of users.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $perPage  = (int) $request->get('per_page', 10);
            $page     = (int) $request->get('page', 1);
            $search   = $request->get('search');
            $roleId   = $request->get('role_id');
            $isActive = $request->get('is_active');

            $query = GmcUser::query()
                ->with([
                    'singleRole',     // default role from gmc_users.Role_Id
                    'roles',
                    'department'         // multiple roles via mapping table
                ])
                ->when($search, function ($q) use ($search) {
                    $q->where(function ($sub) use ($search) {
                        $sub->where('username', 'LIKE', "%{$search}%")
                            ->orWhere('email', 'LIKE', "%{$search}%")
                            ->orWhere('mobile', 'LIKE', "%{$search}%");
                    });
                })
                ->when(!is_null($roleId), function ($q) use ($roleId) {
                    $q->where(function ($sub) use ($roleId) {
                        $sub->where('Role_Id', $roleId) // single role
                            ->orWhereHas('roles', function ($r) use ($roleId) {
                                $r->where('Role_Id', $roleId); // mapped roles
                            });
                    });
                })
                ->when(!is_null($isActive), function ($q) use ($isActive) {
                    $q->where('is_active', (bool) $isActive);
                })
                ->orderByDesc('user_id');

            $users = $query->paginate($perPage, ['*'], 'page', $page);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gandhinagar - Users fetched successfully',
                'data' => $users->items(),
                'meta' => [
                    'current_page'   => $users->currentPage(),
                    'per_page'       => $users->perPage(),
                    'total'          => $users->total(),
                    'last_page'      => $users->lastPage(),
                    'from'           => $users->firstItem(),
                    'to'             => $users->lastItem(),
                    'has_more_pages' => $users->hasMorePages()
                ]
            ], 200);
        } catch (\Exception $e) {
            Log::error('User Index Error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch users. Please try again later.'
            ], 500);
        }
    }


public function getAllUsers(Request $request)
{
    try {
        $search   = $request->get('search');
        $roleId   = $request->get('role_id');
        $isActive = $request->get('is_active');

        $query = GmcUser::query()
            ->with([
                'singleRole',     // default role from gmc_users.Role_Id
                'roles',
                'department',
                     // multiple roles via mapping table
            ])
            ->when($search, function ($q) use ($search) {
                $q->where(function ($sub) use ($search) {
                    $sub->where('username', 'LIKE', "%{$search}%")
                        ->orWhere('email', 'LIKE', "%{$search}%")
                        ->orWhere('mobile', 'LIKE', "%{$search}%");
                });
            })
            ->when(!is_null($roleId), function ($q) use ($roleId) {
                $q->where(function ($sub) use ($roleId) {
                    $sub->where('Role_Id', $roleId) // single role
                        ->orWhereHas('roles', function ($r) use ($roleId) {
                            $r->where('Role_Id', $roleId); // mapped roles
                        });
                });
            })
            ->when(!is_null($isActive), function ($q) use ($isActive) {
                $q->where('is_active', (bool) $isActive);
            })
            ->orderByDesc('user_id');

        // Get all records without pagination
        $users = $query->get();

        return response()->json([
            'success' => true,
            'message' => 'GMC Gandhinagar - All users fetched successfully',
            'data' => $users,
            'meta' => [
                'total_count' => $users->count()
            ]
        ], 200);
    } catch (\Exception $e) {
        Log::error('Get All Users Error', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);

        return response()->json([
            'success' => false,
            'message' => 'Unable to fetch all users. Please try again later.'
        ], 500);
    }
}

public function store(Request $request)
{
    try {
        $validator = Validator::make($request->all(), [
            'username' => 'required|string|max:50|unique:gmc_users,username',
            'email' => 'required|email|max:100|unique:gmc_users,email',
            'mobile' => 'required|string|max:15|unique:gmc_users,mobile',
            'password' => 'required|string|min:8|confirmed',
            'password_confirmation' => 'required|string|min:8',
            'Role_Id' => 'nullable|exists:role_master,Role_Id',
            'department_id'  => 'nullable|exists:main_departments,id',
            'is_active' => 'boolean',
            'session_timeout' => 'integer|min:1|max:1440',
            'password_expiry_days' => 'integer|min:1|max:365',
            'location_id' => 'nullable|exists:locations,location_id'
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        // -----------------------------
        // CREATE USER
        // -----------------------------
        $user = GmcUser::create([
            'username' => $request->username,
            'email' => $request->email,
            'mobile' => $request->mobile,
            'password_hash' => $request->password,
            'Role_Id' => $request->Role_Id,
            'department_id' => $request->department_id,
            'is_active' => $request->boolean('is_active', true),
            'session_timeout' => $request->session_timeout ?? 30,
            'password_expiry_days' => $request->password_expiry_days ?? 90,
            'last_password_change' => now(),
            'created_by' => auth()->id() ?? null
        ]);

        // Check citizen role
        $isCitizen = ($request->Role_Id == 4);

        // -----------------------------
        // CREATE DEVICE QUEUE ENTRY
        // -----------------------------
        $queue = \App\Models\GmcDeviceRegistrationQueue::create([
            'user_id' => $user->user_id,
            'registration_data' => [
                'username' => $request->username,
                'email' => $request->email,
                'mobile' => $request->mobile,
                'Role_Id' => $request->Role_Id,
                'department_id' => $request->department_id,
                'location_id' => $request->location_id,
                'registered_at' => now()->toDateTimeString()
            ],
            'ip_address' => $request->ip() ?? '0.0.0.0',
            'browser_fingerprint' => 'NOT_REQUIRED',
            'operating_system' => $request->header('User-Agent') ?? 'Unknown',
            'browser' => $request->header('User-Agent') ?? 'Unknown',
            'device_name' => 'Unknown',
            'device_type' => 'DESKTOP',
            'department_id' => $request->department_id,
            'location_id' => $request->location_id,
            'status' => $isCitizen ? 'APPROVED' : 'PENDING',
            'is_citizen' => $isCitizen,
            'processed_at' => $isCitizen ? now() : null,
            'processed_by' => $isCitizen ? 1 : null,
            'submitted_at' => now()
        ]);

        // -----------------------------
        // AUTO APPROVE ONLY FOR CITIZEN
        // (NO LOGS HERE)
        // -----------------------------
        if ($isCitizen) {
            $queue->approve(1, 'Auto-approved for citizen registration');
            \App\Models\GmcDeviceApproval::createFromQueue($queue, 1);
        }

        // -----------------------------
        // USER AUDIT LOG (KEEP)
        // -----------------------------
        GmcUserAuditLog::logAction(
            auth()->id() ?? 1,
            'USER_CREATE',
            'Created new user: ' . $user->username .
            ($isCitizen ? ' (Citizen)' : ' (Employee)'),
            $request->ip(),
            $request->header('User-Agent')
        );

        // -----------------------------
        // RESPONSE
        // -----------------------------
        return response()->json([
            'success' => true,
            'message' => 'GMC Gandhinagar - User created successfully',
            'data' => [
                'user_id' => $user->user_id,
                'username' => $user->username,
                'email' => $user->email,
                'Role_Id' => $user->Role_Id,
                'is_active' => $user->is_active,
                'device_status' => $isCitizen ? 'APPROVED' : 'PENDING'
            ]
        ], 201);

    } catch (ValidationException $e) {

        return response()->json([
            'success' => false,
            'message' => 'Validation failed',
            'errors' => $e->errors()
        ], 422);

    } catch (\Exception $e) {

        Log::error('User Create Error: ' . $e->getMessage());

        return response()->json([
            'success' => false,
            'message' => 'Failed to create user.'
        ], 500);
    }
}


    /**
     * Get role name by ID
     */
    private function getRoleName($roleId)
    {
        try {
            $role = RoleMaster::find($roleId);
            return $role ? $role->role_name : 'Unknown';
        } catch (\Exception $e) {
            return 'Unknown';
        }
    }

    /**
     * Create device approval for citizen
     */
    private function createDeviceApproval($userId, $deviceData, $registrationData)
    {
        try {
            // Generate unique device ID
            $deviceUniqueId = md5(
                $deviceData['mac_address'] .
                    $deviceData['hdd_serial'] .
                    $deviceData['browser_fingerprint'] .
                    $userId
            );

            // Default validity: 1 year for citizens
            $validFrom = now();
            $validUntil = now()->addDays(365);

            // Create device approval
            return \App\Models\GmcDeviceApproval::create([
                'user_id' => $userId,
                'device_unique_id' => $deviceUniqueId,
                'device_name' => $deviceData['device_name'] ?? 'Unknown Device',
                'device_type' => $deviceData['device_type'] ?? 'DESKTOP',
                'device_category' => $deviceData['device_category'] ?? 'PRIMARY',
                'ip_address' => $deviceData['ip_address'] ?? '0.0.0.0',
                'mac_address' => $deviceData['mac_address'],
                'hdd_serial' => $deviceData['hdd_serial'],
                'operating_system' => $deviceData['operating_system'],
                'browser' => $deviceData['browser'],
                'browser_fingerprint' => $deviceData['browser_fingerprint'],
                'motherboard_serial' => $deviceData['motherboard_serial'],
                'processor_id' => $deviceData['processor_id'],
                'location_id' => $deviceData['location_id'],
                'department_id' => $deviceData['department_id'],
                'requested_at' => now(),
                'requested_by' => $userId,
                'approved_by' => 1, // Superadmin auto-approval
                'approved_at' => now(),
                'approval_type' => 'AUTO_CITIZEN',
                'valid_from' => $validFrom,
                'valid_until' => $validUntil,
                'status' => 'APPROVED',
                'is_active' => true,
                'is_primary_device' => true
            ]);
        } catch (\Exception $e) {
            Log::error('Create Device Approval Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Display the specified user.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $user = GmcUser::with([
                'singleRole',
                'roles',
                'roleMappings',
                'sessions',
                'auditLogs',
                'department'

            ])->find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'User details fetched successfully',
                'data' => [
                    'user_id' => $user->user_id,
                    'username' => $user->username,
                    'email' => $user->email,
                    'mobile' => $user->mobile,

                    'is_active' => $user->is_active,
                    'is_locked' => $user->is_locked,
                    'locked_reason' => $user->locked_reason,
                    'account_status' => $user->account_status,

                    'mfa_enabled' => $user->mfa_enabled,
                    'password_expired' => $user->isPasswordExpired(),

                    'last_login' => $user->last_login,
                    'days_since_last_login' => $user->days_since_last_login,

                    'created_at' => $user->created_at,
                    'updated_at' => $user->updated_at,

                    // ðŸ”¹ Roles
                    'default_role' => $user->singleRole,
                    'assigned_roles' => $user->roles,

                    // ðŸ”¹ Role mappings
                    'role_mappings' => $user->roleMappings,

                    // ðŸ”¹ Sessions
                    'active_sessions' => $user->activeSessions(),

                    // ðŸ”¹ Audit logs (last 10)
                    'recent_audit_logs' => $user->recentAuditLogs(10),
                ]
            ], 200);
        } catch (\Throwable $e) {

            Log::error('User Show Error', [
                'user_id' => $id,
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch user details'
            ], 500);
        }
    }



    /**
     * Update the specified user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $user = GmcUser::find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found. Please check the user ID.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'username' => 'required|string|max:50|unique:gmc_users,username,' . $id . ',user_id',
                'email' => 'required|email|max:100|unique:gmc_users,email,' . $id . ',user_id',
                'mobile' => 'required|string|max:15|unique:gmc_users,mobile,' . $id . ',user_id',
                'Role_Id' => 'nullable|exists:role_master,Role_Id',
                'department_id' => 'nullable|exists:main_departments,id',
                'is_active' => 'boolean',
                'session_timeout' => 'integer|min:1|max:1440',
                'password_expiry_days' => 'integer|min:1|max:365',
                'is_locked' => 'boolean',
                'locked_reason' => 'nullable|string|max:500',
                'mfa_enabled' => 'boolean'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $updateData = [
                'username' => $request->username,
                'email' => $request->email,
                'mobile' => $request->mobile,
                'Role_Id' => $request->Role_Id,
                'department_id' => $request->department_id,
                'is_active' => $request->boolean('is_active', $user->is_active),
                'session_timeout' => $request->session_timeout ?? $user->session_timeout,
                'password_expiry_days' => $request->password_expiry_days ?? $user->password_expiry_days,
                'is_locked' => $request->boolean('is_locked', $user->is_locked),
                'mfa_enabled' => $request->boolean('mfa_enabled', $user->mfa_enabled),
                'updated_by' => auth()->id() ?? null
            ];

            if ($request->has('locked_reason')) {
                $updateData['locked_reason'] = $request->locked_reason;
            }

            if ($request->boolean('is_locked', $user->is_locked)) {
                $updateData['locked_until'] = now()->addMinutes(30);
            } else {
                $updateData['locked_until'] = null;
                $updateData['locked_reason'] = null;
                $updateData['failed_login_attempts'] = 0;
            }

            $user->update($updateData);

            // Log user update
            GmcUserAuditLog::logAction(
                auth()->id(),
                'USER_UPDATE',
                'Updated user: ' . $user->username,
                $request->ip(),
                $request->header('User-Agent')
            );

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - User updated successfully',
                'data' => $user->refresh()
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('User Update Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to update user. Please check the data and try again.'
            ], 500);
        }
    }
/**
 * Delete user directly without any complications.
 *
 * @param  int  $id
 * @return \Illuminate\Http\JsonResponse
 */
public function destroy($id)
{
    try {
        $user = GmcUser::find($id);

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found.'
            ], 404);
        }

        // Just delete it
        $user->delete();

        return response()->json([
            'success' => true,
            'message' => 'User deleted successfully.',
            'data' => [
                'id' => $id,
                'deleted' => true
            ]
        ], 200);

    } catch (\Exception $e) {
        Log::error('Delete user error: ' . $e->getMessage());

        return response()->json([
            'success' => false,
            'message' => 'Failed to delete user.'
        ], 500);
    }
}

    /**
     * Restore a soft-deleted user.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function restore($id)
    {
        try {
            $user = GmcUser::withTrashed()->find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found. Please check the user ID.'
                ], 404);
            }

            if (!$user->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'User is not deleted.'
                ], 400);
            }

            $user->restore();
            $user->update([
                'deleted_at' => null,
                'deleted_by' => null,
                'deletion_reason' => null,
                'is_active' => true,
                'updated_by' => auth()->id()
            ]);

            // Log user restoration
            GmcUserAuditLog::logAction(
                auth()->id(),
                'USER_RESTORE',
                'Restored user: ' . $user->username,
                request()->ip(),
                request()->header('User-Agent')
            );

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - User restored successfully',
                'data' => $user
            ], 200);
        } catch (\Exception $e) {
            Log::error('User Restore Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to restore user. Please try again later.'
            ], 500);
        }
    }

    /**
     * Get user profile.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function profile()
    {
        try {
            $user = auth()->user();

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not authenticated.'
                ], 401);
            }

            $user->load(['role']);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - User profile retrieved successfully',
                'data' => $user
            ], 200);
        } catch (\Exception $e) {
            Log::error('Profile Fetch Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve profile.'
            ], 500);
        }
    }

    /**
     * Update user profile.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateProfile(Request $request)
    {
        try {
            $user = auth()->user();

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not authenticated.'
                ], 401);
            }

            $validator = Validator::make($request->all(), [
                'email' => 'required|email|max:100|unique:gmc_users,email,' . $user->user_id . ',user_id',
                'mobile' => 'required|string|max:15|unique:gmc_users,mobile,' . $user->user_id . ',user_id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $user->update([
                'email' => $request->email,
                'mobile' => $request->mobile,
                'updated_by' => $user->user_id
            ]);

            // Log profile update
            GmcUserAuditLog::logAction(
                $user->user_id,
                'PROFILE_UPDATE',
                'Updated profile information',
                $request->ip(),
                $request->header('User-Agent')
            );

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Profile updated successfully',
                'data' => $user->refresh()
            ], 200);
        } catch (\Exception $e) {
            Log::error('Profile Update Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to update profile. Please check the data and try again.'
            ], 500);
        }
    }

    /**
     * Toggle MFA for user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function toggleMfa(Request $request, $id)
    {
        try {
            $user = GmcUser::find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found. Please check the user ID.'
                ], 404);
            }

            $mfaEnabled = !$user->mfa_enabled;

            $user->update([
                'mfa_enabled' => $mfaEnabled,
                'mfa_secret' => $mfaEnabled ? $this->generateMfaSecret() : null,
                'updated_by' => auth()->id()
            ]);

            $action = $mfaEnabled ? 'enabled' : 'disabled';

            // Log MFA toggle
            GmcUserAuditLog::logAction(
                auth()->id(),
                'MFA_TOGGLE',
                $action . ' MFA for user: ' . $user->username,
                $request->ip(),
                $request->header('User-Agent')
            );

            return response()->json([
                'success' => true,
                'message' => 'MFA ' . $action . ' successfully for user: ' . $user->username,
                'data' => [
                    'mfa_enabled' => $user->mfa_enabled,
                    'mfa_secret' => $user->mfa_enabled ? $user->mfa_secret : null
                ]
            ], 200);
        } catch (\Exception $e) {
            Log::error('MFA Toggle Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle MFA. Please try again later.'
            ], 500);
        }
    }

    /**
     * Lock/Unlock user account.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function toggleLock(Request $request, $id)
    {
        try {
            $user = GmcUser::find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found. Please check the user ID.'
                ], 404);
            }

            if ($user->user_id === auth()->id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'You cannot lock/unlock your own account.'
                ], 400);
            }

            $isLocked = !$user->is_locked;
            $lockReason = $request->lock_reason ?? ($isLocked ? 'Manually locked by administrator' : null);

            $updateData = [
                'is_locked' => $isLocked,
                'locked_reason' => $lockReason,
                'updated_by' => auth()->id()
            ];

            if ($isLocked) {
                $updateData['locked_until'] = now()->addMinutes(30);
                $updateData['failed_login_attempts'] = 5;
            } else {
                $updateData['locked_until'] = null;
                $updateData['failed_login_attempts'] = 0;
            }

            $user->update($updateData);

            // Invalidate all active sessions if locking
            if ($isLocked) {
                $user->sessions()->where('is_active', true)->update(['is_active' => false]);
            }

            $action = $isLocked ? 'locked' : 'unlocked';

            // Log lock toggle
            GmcUserAuditLog::logAction(
                auth()->id(),
                'ACCOUNT_LOCK_TOGGLE',
                $action . ' account for user: ' . $user->username . ($lockReason ? ' - Reason: ' . $lockReason : ''),
                $request->ip(),
                $request->header('User-Agent')
            );

            return response()->json([
                'success' => true,
                'message' => 'Account ' . $action . ' successfully for user: ' . $user->username,
                'data' => [
                    'is_locked' => $user->is_locked,
                    'locked_until' => $user->locked_until,
                    'locked_reason' => $user->locked_reason
                ]
            ], 200);
        } catch (\Exception $e) {
            Log::error('Account Lock Toggle Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle account lock. Please try again later.'
            ], 500);
        }
    }

    /**
     * Reset user password (Admin function).
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function resetPassword(Request $request, $id)
    {
        try {
            $user = GmcUser::find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found. Please check the user ID.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'new_password' => 'required|string|min:8|confirmed',
                'new_password_confirmation' => 'required|string|min:8'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $user->update([
                'password_hash' => $request->new_password,
                'last_password_change' => now(),
                'failed_login_attempts' => 0,
                'is_locked' => false,
                'locked_until' => null,
                'locked_reason' => null,
                'updated_by' => auth()->id()
            ]);

            // Invalidate all active sessions
            $user->sessions()->where('is_active', true)->update(['is_active' => false]);

            // Log password reset
            GmcUserAuditLog::logAction(
                auth()->id(),
                'PASSWORD_RESET',
                'Reset password for user: ' . $user->username,
                $request->ip(),
                $request->header('User-Agent')
            );

            return response()->json([
                'success' => true,
                'message' => 'Password reset successfully for user: ' . $user->username
            ], 200);
        } catch (\Exception $e) {
            Log::error('Password Reset Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to reset password. Please try again later.'
            ], 500);
        }
    }

    /**
     * Get user sessions.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getUserSessions($id)
    {
        try {
            $user = GmcUser::find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found. Please check the user ID.'
                ], 404);
            }

            $sessions = $user->sessions()
                ->orderBy('login_time', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - User sessions retrieved successfully',
                'data' => $sessions
            ], 200);
        } catch (\Exception $e) {
            Log::error('Get User Sessions Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve user sessions.'
            ], 500);
        }
    }

    /**
     * Get user audit logs.
     *
     * @param  int  $id
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getUserAuditLogs($id, Request $request)
    {
        try {
            $user = GmcUser::find($id);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found. Please check the user ID.'
                ], 404);
            }

            $perPage = $request->get('per_page', 20);
            $page = $request->get('page', 1);
            $actionType = $request->get('action_type', null);
            $dateFrom = $request->get('date_from', null);
            $dateTo = $request->get('date_to', null);

            $query = $user->auditLogs()
                ->when($actionType, function ($query, $actionType) {
                    return $query->where('action_type', $actionType);
                })
                ->when($dateFrom, function ($query, $dateFrom) {
                    return $query->whereDate('performed_at', '>=', $dateFrom);
                })
                ->when($dateTo, function ($query, $dateTo) {
                    return $query->whereDate('performed_at', '<=', $dateTo);
                })
                ->orderBy('performed_at', 'desc');

            $logs = $query->paginate($perPage, ['*'], 'page', $page);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - User audit logs retrieved successfully',
                'data' => $logs->items(),
                'meta' => [
                    'current_page' => $logs->currentPage(),
                    'total' => $logs->total(),
                    'per_page' => $logs->perPage(),
                    'last_page' => $logs->lastPage()
                ]
            ], 200);
        } catch (\Exception $e) {
            Log::error('Get User Audit Logs Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve user audit logs.'
            ], 500);
        }
    }

    /**
     * Generate MFA secret (simplified - use real TOTP in production).
     *
     * @return string
     */
    private function generateMfaSecret()
    {
        // In production, use: https://github.com/antonioribeiro/google2fa
        // For demo, generate random secret
        return 'GMC' . strtoupper(bin2hex(random_bytes(10)));
    }
}
