<?php

namespace App\Http\Controllers;

use App\Models\InstrumentMaster;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class InstrumentMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            // Get query parameters
            $perPage = $request->get('per_page', 10);
            $search = $request->get('search', '');
            $withTrashed = $request->get('with_trashed', false);
            $sortBy = $request->get('sort_by', 'inst_id');
            $sortOrder = $request->get('sort_order', 'desc');

            // Validate sort parameters
            $validSortColumns = ['inst_id', 'inst_type_code', 'inst_type_name', 'created_at', 'updated_at'];
            $sortBy = in_array($sortBy, $validSortColumns) ? $sortBy : 'inst_id';
            $sortOrder = in_array(strtolower($sortOrder), ['asc', 'desc']) ? strtolower($sortOrder) : 'desc';

            // Start query
            $query = InstrumentMaster::query();

            // Include soft deleted records if requested
            if ($withTrashed) {
                $query->withTrashed();
            }

            // Apply search filter
            if ($search) {
                $query->search($search);
            }

            // Apply ordering
            $query->orderBy($sortBy, $sortOrder);

            // Get paginated results
            $instrumentMasters = $query->paginate($perPage);

            // Transform the data
            $transformedData = $instrumentMasters->getCollection()->map(function ($item) {
                return $this->transformInstrument($item);
            });

            $instrumentMasters->setCollection($transformedData);

            return response()->json([
                'success' => true,
                'message' => 'Instrument Masters retrieved successfully.',
                'data' => $instrumentMasters->items(),
                'meta' => [
                    'current_page' => $instrumentMasters->currentPage(),
                    'last_page' => $instrumentMasters->lastPage(),
                    'per_page' => $instrumentMasters->perPage(),
                    'total' => $instrumentMasters->total(),
                    'from' => $instrumentMasters->firstItem(),
                    'to' => $instrumentMasters->lastItem(),
                    'sort_by' => $sortBy,
                    'sort_order' => $sortOrder,
                    'has_more_pages' => $instrumentMasters->hasMorePages()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve Instrument Masters.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        // Validate request
        $validator = Validator::make($request->all(), [
            'inst_type_code' => [
                'nullable',
                'string',
                'max:20',
                Rule::unique('instrument_masters', 'inst_type_code')->whereNull('deleted_at')
            ],
            'inst_type_name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('instrument_masters', 'inst_type_name')->whereNull('deleted_at')
            ]
        ], [
            'inst_type_name.required' => 'Instrument Type Name is required.',
            'inst_type_name.unique' => 'Instrument Type Name already exists.',
            'inst_type_code.unique' => 'Instrument Type Code already exists.',
            'inst_type_code.max' => 'Instrument Type Code must not exceed 20 characters.',
            'inst_type_name.max' => 'Instrument Type Name must not exceed 255 characters.'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Create new instrument master
            $instrumentMaster = InstrumentMaster::create([
                'inst_type_code' => $request->inst_type_code,
                'inst_type_name' => $request->inst_type_name
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Instrument Master created successfully.',
                'data' => $this->transformInstrument($instrumentMaster)
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create Instrument Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): JsonResponse
    {
        try {
            $instrumentMaster = InstrumentMaster::withTrashed()->find($id);

            if (!$instrumentMaster) {
                return response()->json([
                    'success' => false,
                    'message' => 'Instrument Master not found.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Instrument Master retrieved successfully.',
                'data' => $this->transformInstrument($instrumentMaster)
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve Instrument Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $instrumentMaster = InstrumentMaster::find($id);

            if (!$instrumentMaster) {
                return response()->json([
                    'success' => false,
                    'message' => 'Instrument Master not found.'
                ], 404);
            }

            // Validate request
            $validator = Validator::make($request->all(), [
                'inst_type_code' => [
                    'nullable',
                    'string',
                    'max:20',
                    Rule::unique('instrument_masters', 'inst_type_code')
                        ->ignore($id, 'inst_id')
                        ->whereNull('deleted_at')
                ],
                'inst_type_name' => [
                    'required',
                    'string',
                    'max:255',
                    Rule::unique('instrument_masters', 'inst_type_name')
                        ->ignore($id, 'inst_id')
                        ->whereNull('deleted_at')
                ]
            ], [
                'inst_type_name.required' => 'Instrument Type Name is required.',
                'inst_type_name.unique' => 'Instrument Type Name already exists.',
                'inst_type_code.unique' => 'Instrument Type Code already exists.',
                'inst_type_code.max' => 'Instrument Type Code must not exceed 20 characters.',
                'inst_type_name.max' => 'Instrument Type Name must not exceed 255 characters.'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Update instrument master
            $instrumentMaster->update([
                'inst_type_code' => $request->inst_type_code,
                'inst_type_name' => $request->inst_type_name
            ]);

            // Refresh the model to get updated data
            $instrumentMaster->refresh();

            return response()->json([
                'success' => true,
                'message' => 'Instrument Master updated successfully.',
                'data' => $this->transformInstrument($instrumentMaster)
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update Instrument Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $instrumentMaster = InstrumentMaster::find($id);

            if (!$instrumentMaster) {
                return response()->json([
                    'success' => false,
                    'message' => 'Instrument Master not found.'
                ], 404);
            }

            $instrumentMaster->delete();

            return response()->json([
                'success' => true,
                'message' => 'Instrument Master deleted successfully.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete Instrument Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Restore the specified soft deleted resource.
     */
    public function restore(string $id): JsonResponse
    {
        try {
            $instrumentMaster = InstrumentMaster::withTrashed()->find($id);

            if (!$instrumentMaster) {
                return response()->json([
                    'success' => false,
                    'message' => 'Instrument Master not found.'
                ], 404);
            }

            if (!$instrumentMaster->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Instrument Master is not deleted.'
                ], 400);
            }

            // Check for unique constraints before restore
            $existingCode = InstrumentMaster::where('inst_type_code', $instrumentMaster->inst_type_code)
                ->where('inst_id', '!=', $id)
                ->whereNull('deleted_at')
                ->exists();

            if ($existingCode && $instrumentMaster->inst_type_code) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot restore. Instrument Type Code already exists in active records.'
                ], 409);
            }

            $existingName = InstrumentMaster::where('inst_type_name', $instrumentMaster->inst_type_name)
                ->where('inst_id', '!=', $id)
                ->whereNull('deleted_at')
                ->exists();

            if ($existingName) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot restore. Instrument Type Name already exists in active records.'
                ], 409);
            }

            $instrumentMaster->restore();

            return response()->json([
                'success' => true,
                'message' => 'Instrument Master restored successfully.',
                'data' => $this->transformInstrument($instrumentMaster)
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to restore Instrument Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Permanently delete the specified resource.
     */
    public function forceDelete(string $id): JsonResponse
    {
        try {
            $instrumentMaster = InstrumentMaster::withTrashed()->find($id);

            if (!$instrumentMaster) {
                return response()->json([
                    'success' => false,
                    'message' => 'Instrument Master not found.'
                ], 404);
            }

            $instrumentMaster->forceDelete();

            return response()->json([
                'success' => true,
                'message' => 'Instrument Master permanently deleted successfully.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to permanently delete Instrument Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Bulk create Instrument Masters.
     */
    public function bulkStore(Request $request): JsonResponse
    {
        // Validate request
        $validator = Validator::make($request->all(), [
            'instruments' => 'required|array|min:1',
            'instruments.*.inst_type_code' => 'nullable|string|max:20',
            'instruments.*.inst_type_name' => 'required|string|max:255'
        ], [
            'instruments.required' => 'Instruments array is required.',
            'instruments.*.inst_type_name.required' => 'Instrument Type Name is required for all items.',
            'instruments.*.inst_type_code.max' => 'Instrument Type Code must not exceed 20 characters.',
            'instruments.*.inst_type_name.max' => 'Instrument Type Name must not exceed 255 characters.'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $createdInstruments = [];
            $errors = [];

            foreach ($request->instruments as $index => $instrumentData) {
                // Check for duplicate type code if provided
                if (isset($instrumentData['inst_type_code']) && $instrumentData['inst_type_code']) {
                    $existingCode = InstrumentMaster::where('inst_type_code', $instrumentData['inst_type_code'])
                        ->whereNull('deleted_at')
                        ->exists();

                    if ($existingCode) {
                        $errors[] = [
                            'index' => $index,
                            'message' => 'Instrument Type Code already exists: ' . $instrumentData['inst_type_code'],
                            'data' => $instrumentData
                        ];
                        continue;
                    }
                }

                // Check for duplicate type name
                $existingName = InstrumentMaster::where('inst_type_name', $instrumentData['inst_type_name'])
                    ->whereNull('deleted_at')
                    ->exists();

                if ($existingName) {
                    $errors[] = [
                        'index' => $index,
                        'message' => 'Instrument Type Name already exists: ' . $instrumentData['inst_type_name'],
                        'data' => $instrumentData
                    ];
                    continue;
                }

                $instrument = InstrumentMaster::create([
                    'inst_type_code' => $instrumentData['inst_type_code'] ?? null,
                    'inst_type_name' => $instrumentData['inst_type_name']
                ]);

                $createdInstruments[] = $this->transformInstrument($instrument);
            }

            $response = [
                'success' => true,
                'message' => count($createdInstruments) . ' Instrument Master(s) created successfully.',
                'data' => $createdInstruments
            ];

            if (!empty($errors)) {
                $response['errors'] = $errors;
                $response['partial_success'] = count($createdInstruments) > 0;
                $response['message'] .= ' ' . count($errors) . ' item(s) failed due to duplicates.';
            }

            $statusCode = !empty($createdInstruments) ? 201 : 400;

            return response()->json($response, $statusCode);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create Instrument Masters.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get only active (non-deleted) Instrument Masters.
     */
    public function active(Request $request): JsonResponse
    {
        try {
            $search = $request->get('search', '');
            $sortBy = $request->get('sort_by', 'inst_type_name');
            $sortOrder = $request->get('sort_order', 'asc');

            // Validate sort parameters
            $validSortColumns = ['inst_id', 'inst_type_code', 'inst_type_name'];
            $sortBy = in_array($sortBy, $validSortColumns) ? $sortBy : 'inst_type_name';
            $sortOrder = in_array(strtolower($sortOrder), ['asc', 'desc']) ? strtolower($sortOrder) : 'asc';

            $query = InstrumentMaster::active();

            if ($search) {
                $query->search($search);
            }

            $activeInstruments = $query->orderBy($sortBy, $sortOrder)->get();

            $transformedData = $activeInstruments->map(function ($item) {
                return $this->transformInstrument($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Active Instrument Masters retrieved successfully.',
                'data' => $transformedData,
                'meta' => [
                    'total' => $activeInstruments->count(),
                    'sort_by' => $sortBy,
                    'sort_order' => $sortOrder
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve Active Instrument Masters.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Search Instrument Masters by code or name.
     */
    public function search(Request $request): JsonResponse
    {
        try {
            $searchTerm = $request->get('q', '');
            $limit = $request->get('limit', 10);

            if (empty($searchTerm)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Search term is required.',
                    'data' => []
                ], 400);
            }

            $results = InstrumentMaster::active()
                ->search($searchTerm)
                ->limit($limit)
                ->get();

            $transformedData = $results->map(function ($item) {
                return [
                    'inst_id' => $item->inst_id,
                    'inst_type_code' => $item->inst_type_code,
                    'inst_type_name' => $item->inst_type_name,
                    'display_name' => $item->inst_type_code
                        ? $item->inst_type_code . ' - ' . $item->inst_type_name
                        : $item->inst_type_name
                ];
            });

            return response()->json([
                'success' => true,
                'message' => 'Search completed successfully.',
                'data' => $transformedData,
                'meta' => [
                    'search_term' => $searchTerm,
                    'total_results' => $results->count(),
                    'limit' => $limit
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Search failed.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Check if Instrument Type Code exists.
     */
    public function checkCode(Request $request): JsonResponse
    {
        try {
            $code = $request->get('inst_type_code');
            $excludeId = $request->get('exclude_id');

            if (!$code) {
                return response()->json([
                    'success' => false,
                    'message' => 'Instrument Type Code is required.'
                ], 400);
            }

            $query = InstrumentMaster::where('inst_type_code', $code)
                ->whereNull('deleted_at');

            if ($excludeId) {
                $query->where('inst_id', '!=', $excludeId);
            }

            $exists = $query->exists();
            $instrument = $query->first();

            return response()->json([
                'success' => true,
                'exists' => $exists,
                'data' => $exists ? $this->transformInstrument($instrument) : null,
                'message' => $exists
                    ? 'Instrument Type Code already exists.'
                    : 'Instrument Type Code is available.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Check failed.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Check if Instrument Type Name exists.
     */
    public function checkName(Request $request): JsonResponse
    {
        try {
            $name = $request->get('inst_type_name');
            $excludeId = $request->get('exclude_id');

            if (!$name) {
                return response()->json([
                    'success' => false,
                    'message' => 'Instrument Type Name is required.'
                ], 400);
            }

            $query = InstrumentMaster::where('inst_type_name', $name)
                ->whereNull('deleted_at');

            if ($excludeId) {
                $query->where('inst_id', '!=', $excludeId);
            }

            $exists = $query->exists();
            $instrument = $query->first();

            return response()->json([
                'success' => true,
                'exists' => $exists,
                'data' => $exists ? $this->transformInstrument($instrument) : null,
                'message' => $exists
                    ? 'Instrument Type Name already exists.'
                    : 'Instrument Type Name is available.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Check failed.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Transform Instrument data for API response.
     */
    private function transformInstrument($instrument): array
    {
        return [
            'inst_id' => $instrument->inst_id,
            'inst_type_code' => $instrument->inst_type_code,
            'inst_type_name' => $instrument->inst_type_name,
            'display_name' => $instrument->inst_type_code
                ? $instrument->inst_type_code . ' - ' . $instrument->inst_type_name
                : $instrument->inst_type_name,
            'created_at' => $instrument->created_at ? $instrument->created_at->format('Y-m-d H:i:s') : null,
            'updated_at' => $instrument->updated_at ? $instrument->updated_at->format('Y-m-d H:i:s') : null,
            'deleted_at' => $instrument->deleted_at ? $instrument->deleted_at->format('Y-m-d H:i:s') : null,
            'is_deleted' => !is_null($instrument->deleted_at),
            'is_active' => is_null($instrument->deleted_at)
        ];
    }
}
