<?php

namespace App\Http\Controllers;

use App\Models\LedgerDepMapping;
use App\Models\MainDepartment;
use App\Models\LedgerMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LedgerDepMappingController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            $mappings = LedgerDepMapping::with(['mainDepartment'])
                ->get()
                ->map(function ($mapping) {
                    $mapping->all_ledgers = $mapping->ledgers();
                    return $mapping;
                });

            return response()->json([
                'success' => true,
                'data' => $mappings,
                'message' => 'Ledger department mappings retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve mappings.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Dep_Id' => 'nullable|exists:main_departments,id',
            'Ledg_IDs' => 'required|array|min:1', // Changed to required array
            'Ledg_IDs.*' => 'exists:ledger_master,Lg_Id', // Validate each ledger ID
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
                'message' => 'Validation failed.'
            ], 422);
        }

        try {
            // Remove duplicates
            $ledgerIds = array_unique($request->Ledg_IDs);

            $mapping = LedgerDepMapping::create([
                'Dep_Id' => $request->Dep_Id,
                'Ledg_IDs' => $ledgerIds,
            ]);

            $mapping->load(['mainDepartment']);
            $mapping->all_ledgers = $mapping->ledgers();

            return response()->json([
                'success' => true,
                'data' => $mapping,
                'message' => 'Ledger department mapping created successfully.'
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create mapping.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        try {
            $mapping = LedgerDepMapping::with(['mainDepartment'])->find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mapping not found.'
                ], 404);
            }

            $mapping->all_ledgers = $mapping->ledgers();

            return response()->json([
                'success' => true,
                'data' => $mapping,
                'message' => 'Mapping retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve mapping.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $mapping = LedgerDepMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mapping not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Dep_Id' => 'sometimes|exists:main_departments,id',
                'Ledg_IDs' => 'sometimes|array|min:1',
                'Ledg_IDs.*' => 'exists:ledger_master,Lg_Id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed.'
                ], 422);
            }

            $updateData = [];

            if ($request->has('Dep_Id')) {
                $updateData['Dep_Id'] = $request->Dep_Id;
            }

            if ($request->has('Ledg_IDs')) {
                $updateData['Ledg_IDs'] = array_unique($request->Ledg_IDs);
            }

            if (!empty($updateData)) {
                $mapping->update($updateData);
            }

            $mapping->load(['mainDepartment']);
            $mapping->all_ledgers = $mapping->ledgers();

            return response()->json([
                'success' => true,
                'data' => $mapping,
                'message' => 'Mapping updated successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update mapping.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $mapping = LedgerDepMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mapping not found.'
                ], 404);
            }

            $mapping->delete();

            return response()->json([
                'success' => true,
                'message' => 'Mapping deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete mapping.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search mappings by department
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function search(Request $request)
    {
        try {
            $query = LedgerDepMapping::with(['mainDepartment']);

            if ($request->has('Dep_Id')) {
                $query->where('Dep_Id', $request->Dep_Id);
            }

            $results = $query->get()->map(function ($mapping) {
                $mapping->all_ledgers = $mapping->ledgers();
                return $mapping;
            });

            return response()->json([
                'success' => true,
                'data' => $results,
                'message' => 'Search completed successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Search failed.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get mappings by department ID
     *
     * @param  int  $depId
     * @return \Illuminate\Http\Response
     */
    public function byDepartment($depId)
    {
        try {
            $mappings = LedgerDepMapping::with(['mainDepartment'])
                ->where('Dep_Id', $depId)
                ->get()
                ->map(function ($mapping) {
                    $mapping->all_ledgers = $mapping->ledgers();
                    return $mapping;
                });

            return response()->json([
                'success' => true,
                'data' => $mappings,
                'message' => 'Mappings retrieved by department successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve mappings.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get mappings by ledger ID
     *
     * @param  int  $ledgId
     * @return \Illuminate\Http\Response
     */
    public function byLedger($ledgId)
    {
        try {
            $mappings = LedgerDepMapping::with(['mainDepartment'])
                ->whereJsonContains('Ledg_IDs', $ledgId)
                ->get()
                ->map(function ($mapping) {
                    $mapping->all_ledgers = $mapping->ledgers();
                    return $mapping;
                });

            return response()->json([
                'success' => true,
                'data' => $mappings,
                'message' => 'Mappings retrieved by ledger successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve mappings.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all departments for a specific ledger
     *
     * @param  int  $ledgId
     * @return \Illuminate\Http\Response
     */
    public function departmentsForLedger($ledgId)
    {
        try {
            $mappings = LedgerDepMapping::with('mainDepartment')
                ->whereJsonContains('Ledg_IDs', $ledgId)
                ->get();

            $departments = $mappings->pluck('mainDepartment')->filter();

            return response()->json([
                'success' => true,
                'data' => $departments,
                'total_mappings' => $mappings->count(),
                'message' => 'Departments for ledger retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve departments.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all ledgers for a specific department
     *
     * @param  int  $depId
     * @return \Illuminate\Http\Response
     */
    public function ledgersForDepartment($depId)
    {
        try {
            $mappings = LedgerDepMapping::where('Dep_Id', $depId)->get();

            $allLedgerIds = [];
            foreach ($mappings as $mapping) {
                if (!empty($mapping->Ledg_IDs)) {
                    $allLedgerIds = array_merge($allLedgerIds, $mapping->Ledg_IDs);
                }
            }

            $allLedgerIds = array_unique($allLedgerIds);
            $ledgers = LedgerMaster::whereIn('Lg_Id', $allLedgerIds)->get();

            return response()->json([
                'success' => true,
                'data' => $ledgers,
                'total_unique_ledgers' => count($allLedgerIds),
                'message' => 'Ledgers for department retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve ledgers.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Add ledger IDs to an existing mapping
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function addLedgers(Request $request, $id)
    {
        try {
            $mapping = LedgerDepMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mapping not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Ledg_IDs' => 'required|array|min:1',
                'Ledg_IDs.*' => 'exists:ledger_master,Lg_Id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed.'
                ], 422);
            }

            $currentLedgerIds = $mapping->Ledg_IDs ?? [];
            $newLedgerIds = array_unique($request->Ledg_IDs);
            $mergedLedgerIds = array_unique(array_merge($currentLedgerIds, $newLedgerIds));

            $mapping->Ledg_IDs = $mergedLedgerIds;
            $mapping->save();

            $mapping->load(['mainDepartment']);
            $mapping->all_ledgers = $mapping->ledgers();

            return response()->json([
                'success' => true,
                'data' => $mapping,
                'added_count' => count(array_diff($newLedgerIds, $currentLedgerIds)),
                'total_ledgers' => count($mergedLedgerIds),
                'message' => 'Ledgers added successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to add ledgers.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove ledger IDs from an existing mapping
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function removeLedgers(Request $request, $id)
    {
        try {
            $mapping = LedgerDepMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mapping not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Ledg_IDs' => 'required|array|min:1',
                'Ledg_IDs.*' => 'exists:ledger_master,Lg_Id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed.'
                ], 422);
            }

            $currentLedgerIds = $mapping->Ledg_IDs ?? [];
            $removeLedgerIds = $request->Ledg_IDs;
            $updatedLedgerIds = array_diff($currentLedgerIds, $removeLedgerIds);

            // If no ledgers left, delete the mapping
            if (empty($updatedLedgerIds)) {
                $mapping->delete();
                return response()->json([
                    'success' => true,
                    'message' => 'All ledgers removed. Mapping deleted.',
                ]);
            }

            $mapping->Ledg_IDs = array_values($updatedLedgerIds); // Re-index
            $mapping->save();

            $mapping->load(['mainDepartment']);
            $mapping->all_ledgers = $mapping->ledgers();

            return response()->json([
                'success' => true,
                'data' => $mapping,
                'removed_count' => count(array_intersect($currentLedgerIds, $removeLedgerIds)),
                'remaining_ledgers' => count($updatedLedgerIds),
                'message' => 'Ledgers removed successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to remove ledgers.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
