<?php

namespace App\Http\Controllers;

use App\Models\LedgerMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LedgerMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
public function index(Request $request)
{
    try {
        $query = LedgerMaster::with(['subLedgers' => function($query) {
            $query->active()->limit(5);
        }, 'ledgerGroup']);

        // Filter by status if provided
        if ($request->has('status')) {
            $query->where('Lg_Status', $request->boolean('status'));
        }

        // Search functionality
        if ($request->has('search')) {
            $query->search($request->search);
        }

        // Filter by credit limit
        if ($request->has('min_credit_limit')) {
            $query->where('Lg_Cr_Lim', '>=', $request->min_credit_limit);
        }

        if ($request->has('max_credit_limit')) {
            $query->where('Lg_Cr_Lim', '<=', $request->max_credit_limit);
        }

        // Filter by Gr_Id if provided
        if ($request->has('gr_id')) {
            $query->where('Gr_Id', $request->gr_id);
        }

        // New filter 1: ledger_group.is_del === "Y" AND ledger_group.gr_buk === "Y"
        if ($request->has('filter_deleted_buk_y') && $request->boolean('filter_deleted_buk_y')) {
    $query->whereHas('ledgerGroup', function ($q) {
        $q->where('gr_buk', 'Y');
    });
}

        // New filter 2: ledger_group.is_del === "Y" AND ledger_group.gr_buk === "N"
        if ($request->has('filter_deleted_buk_n') && $request->boolean('filter_deleted_buk_n')) {
    $query->whereHas('ledgerGroup', function ($q) {
        $q->where('gr_buk', 'N');
    });
}
        // Sorting
        $sortBy = $request->get('sort_by', 'Lg_Code');
        $sortOrder = $request->get('sort_order', 'asc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->get('per_page', 10);
        $ledgers = $query->paginate($perPage);

        // Format response - Example के अनुसार
        $ledgers->getCollection()->transform(function ($ledger) {
            return [
                'Lg_Id' => $ledger->Lg_Id,
                'Lg_Code' => $ledger->Lg_Code,
                'Lg_Gr' => $ledger->Lg_Gr,
                'Gr_Id' => $ledger->Gr_Id,
                'Lg_Name' => $ledger->Lg_Name,
                'Lg_Address' => $ledger->Lg_Address,
                'Lg_City' => $ledger->Lg_City,
                'Lg_Pin' => $ledger->Lg_Pin,
                'Lg_Phone' => $ledger->Lg_Phone,
                'Lg_Email' => $ledger->Lg_Email,
                'Lg_Sub' => $ledger->Lg_Sub,
                'Lg_Buk' => $ledger->Lg_Buk,
                'Description' => $ledger->Description,
                'Lg_Cr_Lim' => (float) $ledger->Lg_Cr_Lim,
                'Lg_Drw_Pwr' => (float) $ledger->Lg_Drw_Pwr,
                'Lg_Status' => (bool) $ledger->Lg_Status,
                'created_at' => $ledger->created_at,
                'updated_at' => $ledger->updated_at,
                'ledger_group' => $ledger->ledgerGroup ? [
                    'Gr_Id' => $ledger->ledgerGroup->Gr_Id,
                    'Org_Id' => $ledger->ledgerGroup->Org_Id,
                    'gr_code' => $ledger->ledgerGroup->gr_code,
                    'up_gr' => $ledger->ledgerGroup->up_gr,
                    'gr_nm' => $ledger->ledgerGroup->gr_nm,
                    'gr_sr' => $ledger->ledgerGroup->gr_sr,
                    'gr_alt' => $ledger->ledgerGroup->gr_alt,
                    'gr_stt' => $ledger->ledgerGroup->gr_stt,
                    'gr_al_c' => $ledger->ledgerGroup->gr_al_c,
                    'GIFI_Code' => $ledger->ledgerGroup->GIFI_Code,
                    'is_del' => $ledger->ledgerGroup->is_del,
                    'gr_exp' => $ledger->ledgerGroup->gr_exp,
                    'gr_buk' => $ledger->ledgerGroup->gr_buk,
                    'gr_prt' => $ledger->ledgerGroup->gr_prt,
                    'op_id' => $ledger->ledgerGroup->op_id,
                    'dt_tm' => $ledger->ledgerGroup->dt_tm,
                    'naof_grp' => $ledger->ledgerGroup->naof_grp,
                    'credition' => $ledger->ledgerGroup->credition,
                    'eff_gp' => $ledger->ledgerGroup->eff_gp,
                    'eff_sub_led' => $ledger->ledgerGroup->eff_sub_led,
                    'created_at' => $ledger->ledgerGroup->created_at,
                    'updated_at' => $ledger->ledgerGroup->updated_at
                ] : null
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $ledgers,
            'message' => 'Ledgers retrieved successfully'
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to retrieve ledgers',
            'error' => $e->getMessage()
        ], 500);
    }
}

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), LedgerMaster::createRules(), LedgerMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $ledger = LedgerMaster::create($request->all());

            // Load ledgerGroup relationship after creation
            $ledger->load('ledgerGroup');

            return response()->json([
                'success' => true,
                'message' => 'Ledger created successfully',
                'data' => $ledger
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            $ledger = LedgerMaster::with(['subLedgers' => function($query) {
                $query->active();
            }, 'ledgerGroup'])->find($id);

            if (!$ledger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ledger not found'
                ], 404);
            }

            // Format response - Example के अनुसार
            $formattedData = [
                'Lg_Id' => $ledger->Lg_Id,
                'Lg_Code' => $ledger->Lg_Code,
                'Lg_Gr' => $ledger->Lg_Gr,
                'Gr_Id' => $ledger->Gr_Id,
                'Lg_Name' => $ledger->Lg_Name,
                'Lg_Address' => $ledger->Lg_Address,
                'Lg_City' => $ledger->Lg_City,
                'Lg_Pin' => $ledger->Lg_Pin,
                'Lg_Phone' => $ledger->Lg_Phone,
                'Lg_Email' => $ledger->Lg_Email,
                'Lg_Sub' => $ledger->Lg_Sub,
                'Lg_Buk' => $ledger->Lg_Buk,
                'Description' => $ledger->Description,
                'Lg_Cr_Lim' => (float) $ledger->Lg_Cr_Lim,
                'Lg_Drw_Pwr' => (float) $ledger->Lg_Drw_Pwr,
                'Lg_Status' => (bool) $ledger->Lg_Status,
                'created_at' => $ledger->created_at,
                'updated_at' => $ledger->updated_at,
                'sub_ledgers' => $ledger->subLedgers,
                'ledger_group' => $ledger->ledgerGroup ? [
                    'Gr_Id' => $ledger->ledgerGroup->Gr_Id,
                    'Org_Id' => $ledger->ledgerGroup->Org_Id,
                    'gr_code' => $ledger->ledgerGroup->gr_code,
                    'up_gr' => $ledger->ledgerGroup->up_gr,
                    'gr_nm' => $ledger->ledgerGroup->gr_nm,
                    'gr_sr' => $ledger->ledgerGroup->gr_sr,
                    'gr_alt' => $ledger->ledgerGroup->gr_alt,
                    'gr_stt' => $ledger->ledgerGroup->gr_stt,
                    'gr_al_c' => $ledger->ledgerGroup->gr_al_c,
                    'GIFI_Code' => $ledger->ledgerGroup->GIFI_Code,
                    'is_del' => $ledger->ledgerGroup->is_del,
                    'gr_exp' => $ledger->ledgerGroup->gr_exp,
                    'gr_buk' => $ledger->ledgerGroup->gr_buk,
                    'gr_prt' => $ledger->ledgerGroup->gr_prt,
                    'op_id' => $ledger->ledgerGroup->op_id,
                    'dt_tm' => $ledger->ledgerGroup->dt_tm,
                    'naof_grp' => $ledger->ledgerGroup->naof_grp,
                    'credition' => $ledger->ledgerGroup->credition,
                    'eff_gp' => $ledger->ledgerGroup->eff_gp,
                    'eff_sub_led' => $ledger->ledgerGroup->eff_sub_led,
                    'created_at' => $ledger->ledgerGroup->created_at,
                    'updated_at' => $ledger->ledgerGroup->updated_at
                ] : null
            ];

            return response()->json([
                'success' => true,
                'data' => $formattedData,
                'message' => 'Ledger retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $ledger = LedgerMaster::find($id);

            if (!$ledger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ledger not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), LedgerMaster::updateRules($id), LedgerMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $ledger->update($request->all());

            // Refresh the model to get updated data
            $ledger->refresh();
            // Load ledgerGroup relationship
            $ledger->load('ledgerGroup');

            return response()->json([
                'success' => true,
                'message' => 'Ledger updated successfully',
                'data' => $ledger
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $ledger = LedgerMaster::find($id);

            if (!$ledger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ledger not found'
                ], 404);
            }

            if (!$ledger->canBeDeleted()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete ledger as it has sub-ledgers'
                ], 400);
            }

            $ledger->delete();

            return response()->json([
                'success' => true,
                'message' => 'Ledger deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Deactivate a ledger.
     */
    public function deactivate($id)
    {
        try {
            $ledger = LedgerMaster::with('ledgerGroup')->find($id); // ledgerGroup load किया

            if (!$ledger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ledger not found'
                ], 404);
            }

            $ledger->deactivate();

            return response()->json([
                'success' => true,
                'message' => 'Ledger deactivated successfully',
                'data' => $ledger
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to deactivate ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Activate a ledger.
     */
    public function activate($id)
    {
        try {
            $ledger = LedgerMaster::with('ledgerGroup')->find($id); // ledgerGroup load किया

            if (!$ledger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ledger not found'
                ], 404);
            }

            $ledger->activate();

            return response()->json([
                'success' => true,
                'message' => 'Ledger activated successfully',
                'data' => $ledger
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to activate ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get ledgers with their sub-ledgers count.
     */
    public function getLedgersWithCount(Request $request)
    {
        try {
            $query = LedgerMaster::withCount('subLedgers')->with('ledgerGroup'); // ledgerGroup load किया

            if ($request->has('active_only')) {
                $query->active();
            }

            if ($request->has('gr_id')) {
                $query->where('Gr_Id', $request->gr_id);
            }

            $ledgers = $query->get();

            return response()->json([
                'success' => true,
                'data' => $ledgers,
                'message' => 'Ledgers with sub-ledgers count retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve ledgers with count',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk create ledgers.
     */
    public function bulkStore(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'ledgers' => 'required|array|min:1',
                'ledgers.*.Lg_Code' => 'required|string|max:50',
                'ledgers.*.Lg_Name' => 'required|string|max:200',
                'ledgers.*.Lg_Gr' => 'nullable|string|max:100',
                'ledgers.*.Gr_Id' => 'nullable|integer|exists:ledg_grps,Gr_Id',
                'ledgers.*.Lg_Address' => 'nullable|string',
                'ledgers.*.Lg_City' => 'nullable|string|max:100',
                'ledgers.*.Lg_Phone' => 'nullable|string|max:20',
                'ledgers.*.Lg_Email' => 'nullable|email|max:100',
                'ledgers.*.Description' => 'nullable|string|max:255',
                'ledgers.*.Lg_Cr_Lim' => 'nullable|numeric|min:0',
                'ledgers.*.Lg_Drw_Pwr' => 'nullable|numeric|min:0',
                'ledgers.*.Lg_Status' => 'nullable|boolean'
            ], LedgerMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $createdLedgers = [];
            $errors = [];

            foreach ($request->ledgers as $index => $ledgerData) {
                try {
                    // Check for duplicate code
                    $exists = LedgerMaster::where('Lg_Code', $ledgerData['Lg_Code'])->exists();
                    if ($exists) {
                        $errors[] = [
                            'index' => $index,
                            'message' => 'Ledger code ' . $ledgerData['Lg_Code'] . ' already exists'
                        ];
                        continue;
                    }

                    $ledger = LedgerMaster::create($ledgerData);
                    $ledger->load('ledgerGroup'); // ledgerGroup relationship load किया
                    $createdLedgers[] = $ledger;

                } catch (\Exception $e) {
                    $errors[] = [
                        'index' => $index,
                        'message' => $e->getMessage()
                    ];
                }
            }

            return response()->json([
                'success' => true,
                'message' => count($createdLedgers) . ' ledgers created successfully',
                'data' => $createdLedgers,
                'errors' => $errors,
                'total_created' => count($createdLedgers),
                'total_failed' => count($errors)
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create bulk ledgers',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
