<?php

namespace App\Http\Controllers;

use App\Models\MainDepartment;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Throwable;

class MainDepartmentController extends Controller
{
    // GET all
    public function index(Request $request)
    {
        try {
            $queryValue = $request->input('query');
            $isvisible = $request->input('isvisible');

            $departments = MainDepartment::with('category', 'parentDepartment');

            if ($queryValue) {
                $departments->where(function ($q) use ($queryValue) {
                    $q->where('Upgrp', 'like', $queryValue . '%');
                });
            }

            // Filter by isvisible status if provided
            if ($isvisible !== null) {
                $departments->where('isvisible', $isvisible);
            }

            $data = $departments->get();

            return response()->json([
                'status' => 'success',
                'message' => 'Main departments fetched successfully',
                'data' => $data
            ]);
        } catch (\Throwable $e) {
            Log::error('Fetch Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch main departments',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    // GET one
    public function show($id)
    {
        try {
            $data = MainDepartment::with('category')->find($id);

            if (!$data) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Main department not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Main department fetched successfully',
                'data' => $data
            ]);
        } catch (Throwable $e) {
            Log::error('Show Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch main department',
                'error' => $e->getMessage()
            ], 500);
        }
    }

   // CREATE
public function store(Request $request)
{
    try {
        // Default Grsh
        $grsh = $request->has('Grsh') ? $request->input('Grsh') : '1';
        $grsrn = $request->has('Grsm')
            ? $request->input('Grsm')
            : MainDepartment::where('Grsh', $grsh)->count() + 1;

        // Format code for 10 pairs (20 digits)
       
        $prefix = str_pad($grsrn, 2, '0', STR_PAD_LEFT);
        $codeFormat = $prefix . str_repeat('000000', 3); // 2 + (6*3) = 20 digits
     
        // Alternative approach:
        $codeFormat = str_pad($prefix, 20, '0', STR_PAD_RIGHT);

        $data = [
            'desc'        => $request->input('desc'),
            'category_id' => $request->input('category_id'),
            'func_id'     => $request->input('func_id'),
            'Grsh'        => $grsh,
            'Grsm'        => (string) $grsrn,
            'isvisible'   => $request->input('isvisible', 1),
            'Upgrp'       => $request->input('Upgrp', $codeFormat),
            'Grcod'       => $request->input('Grcod', $codeFormat),
            'Menu_Links'  => $request->input('Menu_Links'),
        ];

        $department = MainDepartment::create($data);

        return response()->json([
            'status' => 'success',
            'message' => 'Main department created successfully',
            'data' => $department
        ], 201);
    } catch (ValidationException $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Validation failed',
            'errors' => $e->errors()
        ], 422);
    } catch (Throwable $e) {
        Log::error('Create Error: ' . $e->getMessage());
        return response()->json([
            'status' => 'error',
            'message' => 'Unable to create main department',
            'error' => $e->getMessage()
        ], 500);
    }
}

    // UPDATE
    public function update(Request $request, $id)
    {
        try {
            $department = MainDepartment::find($id);

            if (!$department) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Main department not found'
                ], 404);
            }

            $data = $request->validate([
                'desc'        => 'sometimes|nullable|string',
                'category_id' => 'sometimes|nullable|exists:categories,category_id',
                'Upgrp'       => 'sometimes|nullable|string',
                'Grcod'       => 'sometimes|nullable|string',
                'Grsm'        => 'sometimes|nullable|string',
                'Grsh'        => 'sometimes|nullable|string',
                'isvisible'   => 'sometimes|nullable|integer|in:0,1',
                'func_id'     => 'sometimes|nullable|exists:main_departments,id',
                'Menu_Links'  => 'sometimes|nullable|string',
            ]);

            $department->update($data);

            return response()->json([
                'status' => 'success',
                'message' => 'Main department updated successfully',
                'data' => $department
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('Update Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to update main department',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // TOGGLE VISIBILITY
    public function toggleVisibility($id)
    {
        try {
            $department = MainDepartment::find($id);

            if (!$department) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Main department not found'
                ], 404);
            }

            // Toggle between 0 and 1
            $department->update([
                'isvisible' => $department->isvisible == 1 ? 0 : 1
            ]);

            $status = $department->isvisible == 1 ? 'Active' : 'Inactive';

            return response()->json([
                'status' => 'success',
                'message' => 'Main department visibility toggled successfully',
                'data' => [
                    'id' => $department->id,
                    'isvisible' => $department->isvisible,
                    'status' => $status
                ]
            ]);
        } catch (Throwable $e) {
            Log::error('Toggle Visibility Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to toggle main department visibility',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // DELETE
    public function destroy($id)
    {
        try {
            $department = MainDepartment::find($id);

            if (!$department) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Main department not found'
                ], 404);
            }

            $department->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'Main department deleted successfully'
            ]);
        } catch (Throwable $e) {
            Log::error('Delete Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to delete main department',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
