<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Module;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class ModuleController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        try {
            $query = Module::query();

            // Filter by is_superadmin
            if ($request->has('is_superadmin')) {
                $query->where('is_superadmin', $request->boolean('is_superadmin'));
            }

            // Search functionality
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('module_name', 'LIKE', "%{$search}%")
                      ->orWhere('description', 'LIKE', "%{$search}%");
                });
            }

            // Load relationships if requested
            if ($request->boolean('with_permissions')) {
                $query->with('permissions');
            }

            $perPage = $request->get('per_page', 15);
            $modules = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $modules,
                'message' => 'Modules retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve modules',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'module_name' => 'required|string|max:191',
            'description' => 'nullable|string|max:191',
            'is_superadmin' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $module = Module::create($request->all());

            return response()->json([
                'success' => true,
                'data' => $module,
                'message' => 'Module created successfully'
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create module',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id): JsonResponse
    {
        try {
            $module = Module::with('permissions')->find($id);

            if (!$module) {
                return response()->json([
                    'success' => false,
                    'message' => 'Module not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $module,
                'message' => 'Module retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve module',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'module_name' => 'string|max:191',
            'description' => 'nullable|string|max:191',
            'is_superadmin' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $module = Module::find($id);

            if (!$module) {
                return response()->json([
                    'success' => false,
                    'message' => 'Module not found'
                ], 404);
            }

            $module->update($request->all());

            return response()->json([
                'success' => true,
                'data' => $module,
                'message' => 'Module updated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update module',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            $module = Module::find($id);

            if (!$module) {
                return response()->json([
                    'success' => false,
                    'message' => 'Module not found'
                ], 404);
            }

            // Check if module has permissions
            if ($module->permissions()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete module. It has associated permissions.'
                ], 400);
            }

            $module->delete();

            return response()->json([
                'success' => true,
                'message' => 'Module deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete module',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
