<?php

namespace App\Http\Controllers;

use App\Models\NewTenderWork;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class NewTenderWorkController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            // Get query parameters
            $perPage = $request->get('per_page', 10);
            $search = $request->get('search', '');
            $withTrashed = $request->get('with_trashed', false);

            // Start query
            $query = NewTenderWork::query();

            // Include soft deleted records if requested
            if ($withTrashed) {
                $query->withTrashed();
            }

            // Apply search filter
            if ($search) {
                $query->where(function ($q) use ($search) {
                    $q->where('work_code', 'like', "%{$search}%")
                        ->orWhere('work_name', 'like', "%{$search}%");
                });
            }

            // Apply ordering
            $query->orderBy('tend_work_id', 'desc');

            // Get paginated results
            $newTenderWorks = $query->paginate($perPage);

            // Transform the data
            $transformedData = $newTenderWorks->getCollection()->map(function ($item) {
                return $this->transformTenderWork($item);
            });

            $newTenderWorks->setCollection($transformedData);

            return response()->json([
                'success' => true,
                'message' => 'New Tender Works retrieved successfully.',
                'data' => $newTenderWorks->items(),
                'meta' => [
                    'current_page' => $newTenderWorks->currentPage(),
                    'last_page' => $newTenderWorks->lastPage(),
                    'per_page' => $newTenderWorks->perPage(),
                    'total' => $newTenderWorks->total(),
                    'from' => $newTenderWorks->firstItem(),
                    'to' => $newTenderWorks->lastItem(),
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve New Tender Works.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    public function getAllWithoutPagination(Request $request): JsonResponse
    {
        try {
            // Get query parameters
            $search = $request->get('search', '');
            $withTrashed = $request->get('with_trashed', false);

            // Start query
            $query = NewTenderWork::query();

            // Include soft deleted records if requested
            if ($withTrashed) {
                $query->withTrashed();
            }

            // Apply search filter
            if ($search) {
                $query->where(function ($q) use ($search) {
                    $q->where('work_code', 'like', "%{$search}%")
                        ->orWhere('work_name', 'like', "%{$search}%");
                });
            }

            // Apply ordering
            $query->orderBy('tend_work_id', 'desc');

            // Get all results (without pagination)
            $newTenderWorks = $query->get();

            // Transform the data
            $transformedData = $newTenderWorks->map(function ($item) {
                return $this->transformTenderWork($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'All New Tender Works retrieved successfully.',
                'data' => $transformedData,
                'meta' => [
                    'total_count' => $transformedData->count()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve New Tender Works.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        // Validate request
        $validator = Validator::make($request->all(), [
            'work_code' => 'nullable|string|max:20',
            'work_name' => 'required|string|max:255'
        ], [
            'work_name.required' => 'Work Name is required.',
            'work_code.max' => 'Work Code must not exceed 20 characters.',
            'work_name.max' => 'Work Name must not exceed 255 characters.'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Create new tender work
            $newTenderWork = NewTenderWork::create([
                'work_code' => $request->work_code,
                'work_name' => $request->work_name
            ]);

            return response()->json([
                'success' => true,
                'message' => 'New Tender Work created successfully.',
                'data' => $this->transformTenderWork($newTenderWork)
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create New Tender Work.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): JsonResponse
    {
        try {
            $newTenderWork = NewTenderWork::withTrashed()->find($id);

            if (!$newTenderWork) {
                return response()->json([
                    'success' => false,
                    'message' => 'New Tender Work not found.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'New Tender Work retrieved successfully.',
                'data' => $this->transformTenderWork($newTenderWork)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve New Tender Work.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $newTenderWork = NewTenderWork::find($id);

            if (!$newTenderWork) {
                return response()->json([
                    'success' => false,
                    'message' => 'New Tender Work not found.'
                ], 404);
            }

            // Validate request
            $validator = Validator::make($request->all(), [
                'work_code' => 'nullable|string|max:20',
                'work_name' => 'sometimes|required|string|max:255'
            ], [
                'work_name.required' => 'Work Name is required.',
                'work_code.max' => 'Work Code must not exceed 20 characters.',
                'work_name.max' => 'Work Name must not exceed 255 characters.'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Update only provided fields
            $updateData = [];
            if ($request->has('work_code')) {
                $updateData['work_code'] = $request->work_code;
            }
            if ($request->has('work_name')) {
                $updateData['work_name'] = $request->work_name;
            }

            $newTenderWork->update($updateData);

            // Refresh the model to get updated data
            $newTenderWork->refresh();

            return response()->json([
                'success' => true,
                'message' => 'New Tender Work updated successfully.',
                'data' => $this->transformTenderWork($newTenderWork)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update New Tender Work.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $newTenderWork = NewTenderWork::find($id);

            if (!$newTenderWork) {
                return response()->json([
                    'success' => false,
                    'message' => 'New Tender Work not found.'
                ], 404);
            }

            $newTenderWork->delete();

            return response()->json([
                'success' => true,
                'message' => 'New Tender Work deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete New Tender Work.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Restore the specified soft deleted resource.
     */
    public function restore(string $id): JsonResponse
    {
        try {
            $newTenderWork = NewTenderWork::withTrashed()->find($id);

            if (!$newTenderWork) {
                return response()->json([
                    'success' => false,
                    'message' => 'New Tender Work not found.'
                ], 404);
            }

            if (!$newTenderWork->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'New Tender Work is not deleted.'
                ], 400);
            }

            $newTenderWork->restore();

            return response()->json([
                'success' => true,
                'message' => 'New Tender Work restored successfully.',
                'data' => $this->transformTenderWork($newTenderWork)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to restore New Tender Work.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Permanently delete the specified resource.
     */
    public function forceDelete(string $id): JsonResponse
    {
        try {
            $newTenderWork = NewTenderWork::withTrashed()->find($id);

            if (!$newTenderWork) {
                return response()->json([
                    'success' => false,
                    'message' => 'New Tender Work not found.'
                ], 404);
            }

            $newTenderWork->forceDelete();

            return response()->json([
                'success' => true,
                'message' => 'New Tender Work permanently deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to permanently delete New Tender Work.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Transform tender work data for API response.
     */
    private function transformTenderWork($tenderWork): array
    {
        return [
            'tend_work_id' => $tenderWork->tend_work_id,
            'work_code' => $tenderWork->work_code,
            'work_name' => $tenderWork->work_name,
            'created_at' => $tenderWork->created_at ? $tenderWork->created_at->format('Y-m-d H:i:s') : null,
            'updated_at' => $tenderWork->updated_at ? $tenderWork->updated_at->format('Y-m-d H:i:s') : null,
            'deleted_at' => $tenderWork->deleted_at ? $tenderWork->deleted_at->format('Y-m-d H:i:s') : null,
            'is_deleted' => !is_null($tenderWork->deleted_at)
        ];
    }

    /**
     * Bulk create new tender works.
     */
    public function bulkStore(Request $request): JsonResponse
    {
        // Validate request
        $validator = Validator::make($request->all(), [
            'works' => 'required|array|min:1',
            'works.*.work_code' => 'nullable|string|max:20',
            'works.*.work_name' => 'required|string|max:255'
        ], [
            'works.required' => 'Works array is required.',
            'works.*.work_name.required' => 'Work Name is required for all items.',
            'works.*.work_code.max' => 'Work Code must not exceed 20 characters.',
            'works.*.work_name.max' => 'Work Name must not exceed 255 characters.'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $createdWorks = [];

            foreach ($request->works as $workData) {
                $newTenderWork = NewTenderWork::create([
                    'work_code' => $workData['work_code'] ?? null,
                    'work_name' => $workData['work_name']
                ]);

                $createdWorks[] = $this->transformTenderWork($newTenderWork);
            }

            return response()->json([
                'success' => true,
                'message' => count($createdWorks) . ' New Tender Work(s) created successfully.',
                'data' => $createdWorks
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create New Tender Works.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get only active (non-deleted) tender works.
     */
    public function active(): JsonResponse
    {
        try {
            $activeWorks = NewTenderWork::orderBy('work_name', 'asc')->get();

            $transformedData = $activeWorks->map(function ($item) {
                return $this->transformTenderWork($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Active New Tender Works retrieved successfully.',
                'data' => $transformedData
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve Active New Tender Works.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
