<?php

namespace App\Http\Controllers;

use App\Models\OrganisationFinYear;
use App\Models\OrganisationSetting;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class OrganisationFinYearController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            // Build query with filters
            $query = OrganisationFinYear::query();

            // Eager load organization relationship
            if ($request->has('include_organisation') && $request->include_organisation == 'true') {
                $query->with('organisation');
            }

            // Apply filters
            if ($request->has('org_id') && !empty($request->org_id)) {
                $query->where('Org_Id', $request->org_id);
            }

            if ($request->has('status') && !empty($request->status)) {
                $query->where('Fin_Status', $request->status);
            }

            if ($request->has('year') && !empty($request->year)) {
                $query->where('Fin_Year', 'like', '%' . $request->year . '%');
            }

            // Date range filters
            if ($request->has('from_date')) {
                $query->where('Fin_Ed_Dt', '>=', $request->from_date);
            }

            if ($request->has('to_date')) {
                $query->where('Fin_St_Dt', '<=', $request->to_date);
            }

            // Search functionality
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('Fin_Year', 'like', "%{$search}%")
                        ->orWhereHas('organisation', function ($q) use ($search) {
                            $q->where('Org_Code', 'like', "%{$search}%")
                              ->orWhere('Org_Name', 'like', "%{$search}%");
                        });
                });
            }

            // Sorting
            $sortField = $request->get('sort_by', 'Fin_St_Dt');
            $sortDirection = $request->get('sort_dir', 'desc');

            // Validate sort field
            $allowedSortFields = ['Org_FYID', 'Org_Id', 'Fin_Year', 'Fin_St_Dt', 'Fin_Ed_Dt', 'Fin_Status', 'created_at'];
            if (!in_array($sortField, $allowedSortFields)) {
                $sortField = 'Fin_St_Dt';
            }

            // Validate sort direction
            $sortDirection = strtolower($sortDirection) === 'asc' ? 'asc' : 'desc';

            $query->orderBy($sortField, $sortDirection);

            // Pagination
            $perPage = $request->get('per_page', 15);
            $perPage = min(max(1, $perPage), 100);

            $finYears = $query->paginate($perPage);

            // Add organization details if not eager loaded
            if (!$request->has('include_organisation') || $request->include_organisation != 'true') {
                $finYears->getCollection()->transform(function ($finYear) {
                    $finYear->organization_name = $finYear->organization_name;
                    $finYear->organization_code = $finYear->organization_code;
                    return $finYear;
                });
            }

            return response()->json([
                'success' => true,
                'message' => 'Financial years retrieved successfully',
                'data' => $finYears
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve financial years',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        DB::beginTransaction();

        try {
            // Validation rules
            $validator = Validator::make($request->all(), [
                'Org_Id' => 'required|integer|exists:organisation_settings,Org_Id',
                'Fin_Year' => [
                    'required',
                    'string',
                    'max:25',
                    function ($attribute, $value, $fail) use ($request) {
                        $exists = OrganisationFinYear::where('Org_Id', $request->Org_Id)
                            ->where('Fin_Year', $value)
                            ->exists();

                        if ($exists) {
                            $fail('This financial year already exists for the selected organization.');
                        }
                    }
                ],
                'Fin_St_Dt' => 'required|date|date_format:Y-m-d',
                'Fin_Ed_Dt' => 'required|date|date_format:Y-m-d|after:Fin_St_Dt',
                'Fin_Status' => 'nullable|in:Active,Inactive,Closed'
            ], [
                'Org_Id.required' => 'Organization ID is required',
                'Org_Id.integer' => 'Organization ID must be an integer',
                'Org_Id.exists' => 'The selected organization does not exist',
                'Fin_Year.required' => 'Financial year is required',
                'Fin_Year.max' => 'Financial year cannot exceed 25 characters',
                'Fin_St_Dt.required' => 'Start date is required',
                'Fin_St_Dt.date' => 'Start date must be a valid date',
                'Fin_St_Dt.date_format' => 'Start date must be in YYYY-MM-DD format',
                'Fin_Ed_Dt.required' => 'End date is required',
                'Fin_Ed_Dt.date' => 'End date must be a valid date',
                'Fin_Ed_Dt.date_format' => 'End date must be in YYYY-MM-DD format',
                'Fin_Ed_Dt.after' => 'End date must be after start date',
                'Fin_Status.in' => 'Status must be one of: Active, Inactive, Closed'
            ]);

            // Additional validation for date range
            $validator->after(function ($validator) use ($request) {
                if ($request->Fin_St_Dt && $request->Fin_Ed_Dt) {
                    $start = Carbon::parse($request->Fin_St_Dt);
                    $end = Carbon::parse($request->Fin_Ed_Dt);

                    // Check if financial year exceeds 2 years
                    if ($start->diffInYears($end) > 2) {
                        $validator->errors()->add('Fin_Ed_Dt', 'Financial year cannot exceed 2 years.');
                    }

                    // Check if dates overlap with existing financial years
                    $overlappingYear = OrganisationFinYear::where('Org_Id', $request->Org_Id)
                        ->where(function ($query) use ($request) {
                            $query->whereBetween('Fin_St_Dt', [$request->Fin_St_Dt, $request->Fin_Ed_Dt])
                                ->orWhereBetween('Fin_Ed_Dt', [$request->Fin_St_Dt, $request->Fin_Ed_Dt])
                                ->orWhere(function ($query) use ($request) {
                                    $query->where('Fin_St_Dt', '<=', $request->Fin_St_Dt)
                                        ->where('Fin_Ed_Dt', '>=', $request->Fin_Ed_Dt);
                                });
                        })
                        ->first();

                    if ($overlappingYear) {
                        $validator->errors()->add('Fin_Year', 'Financial year dates overlap with existing financial year: ' . $overlappingYear->Fin_Year);
                    }

                    // Optional: Validate financial year format (typically April to March)
                    /*
                    if ($start->day != 1 || $start->month != 4) {
                        $validator->errors()->add('Fin_St_Dt', 'Financial year should typically start on 1st April.');
                    }

                    if (!($end->day == 31 && $end->month == 3)) {
                        $validator->errors()->add('Fin_Ed_Dt', 'Financial year should typically end on 31st March.');
                    }
                    */
                }
            });

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], Response::HTTP_BAD_REQUEST);
            }

            // Prepare data
            $data = $validator->validated();
            $data['Fin_Status'] = $data['Fin_Status'] ?? 'Active';

            // Create financial year
            $finYear = OrganisationFinYear::create($data);

            // If this is set as active, deactivate other active years for the same organization
            if ($finYear->Fin_Status === 'Active') {
                OrganisationFinYear::where('Org_Id', $finYear->Org_Id)
                    ->where('Org_FYID', '!=', $finYear->Org_FYID)
                    ->where('Fin_Status', 'Active')
                    ->update(['Fin_Status' => 'Inactive']);
            }

            DB::commit();

            // Load organization relationship
            $finYear->load('organisation');

            return response()->json([
                'success' => true,
                'message' => 'Financial year created successfully',
                'data' => $finYear
            ], Response::HTTP_CREATED);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to create financial year',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            // Validate ID
            if (!is_numeric($id) || $id <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid financial year ID'
                ], Response::HTTP_BAD_REQUEST);
            }

            $finYear = OrganisationFinYear::with('organisation')->find($id);

            if (!$finYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year not found'
                ], Response::HTTP_NOT_FOUND);
            }

            // Add calculated fields
            $finYear->duration_days = $finYear->getDurationInDays();
            $finYear->duration_months = $finYear->getDurationInMonths();
            $finYear->is_current = $finYear->isDateInRange(now());

            return response()->json([
                'success' => true,
                'message' => 'Financial year retrieved successfully',
                'data' => $finYear
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve financial year',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        DB::beginTransaction();

        try {
            // Validate ID
            if (!is_numeric($id) || $id <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid financial year ID'
                ], Response::HTTP_BAD_REQUEST);
            }

            $finYear = OrganisationFinYear::find($id);

            if (!$finYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year not found'
                ], Response::HTTP_NOT_FOUND);
            }

            // Validation rules
            $validator = Validator::make($request->all(), [
                'Org_Id' => 'sometimes|required|integer|exists:organisation_settings,Org_Id',
                'Fin_Year' => [
                    'sometimes',
                    'required',
                    'string',
                    'max:25',
                    function ($attribute, $value, $fail) use ($request, $id, $finYear) {
                        $orgId = $request->Org_Id ?? $finYear->Org_Id;
                        $exists = OrganisationFinYear::where('Org_Id', $orgId)
                            ->where('Fin_Year', $value)
                            ->where('Org_FYID', '!=', $id)
                            ->exists();

                        if ($exists) {
                            $fail('This financial year already exists for the selected organization.');
                        }
                    }
                ],
                'Fin_St_Dt' => 'sometimes|required|date|date_format:Y-m-d',
                'Fin_Ed_Dt' => 'sometimes|required|date|date_format:Y-m-d|after:Fin_St_Dt',
                'Fin_Status' => 'nullable|in:Active,Inactive,Closed'
            ], [
                'Org_Id.integer' => 'Organization ID must be an integer',
                'Org_Id.exists' => 'The selected organization does not exist',
                'Fin_Year.max' => 'Financial year cannot exceed 25 characters',
                'Fin_St_Dt.date' => 'Start date must be a valid date',
                'Fin_St_Dt.date_format' => 'Start date must be in YYYY-MM-DD format',
                'Fin_Ed_Dt.date' => 'End date must be a valid date',
                'Fin_Ed_Dt.date_format' => 'End date must be in YYYY-MM-DD format',
                'Fin_Ed_Dt.after' => 'End date must be after start date',
                'Fin_Status.in' => 'Status must be one of: Active, Inactive, Closed'
            ]);

            // Additional validation for date range
            $validator->after(function ($validator) use ($request, $id, $finYear) {
                // Check for date overlaps if dates are being updated
                if ($request->has('Fin_St_Dt') || $request->has('Fin_Ed_Dt')) {
                    $startDate = $request->Fin_St_Dt ?? $finYear->Fin_St_Dt;
                    $endDate = $request->Fin_Ed_Dt ?? $finYear->Fin_Ed_Dt;
                    $orgId = $request->Org_Id ?? $finYear->Org_Id;

                    $overlappingYear = OrganisationFinYear::where('Org_Id', $orgId)
                        ->where('Org_FYID', '!=', $id)
                        ->where(function ($query) use ($startDate, $endDate) {
                            $query->whereBetween('Fin_St_Dt', [$startDate, $endDate])
                                ->orWhereBetween('Fin_Ed_Dt', [$startDate, $endDate])
                                ->orWhere(function ($query) use ($startDate, $endDate) {
                                    $query->where('Fin_St_Dt', '<=', $startDate)
                                        ->where('Fin_Ed_Dt', '>=', $endDate);
                                });
                        })
                        ->first();

                    if ($overlappingYear) {
                        $validator->errors()->add('Fin_Year', 'Financial year dates overlap with existing financial year: ' . $overlappingYear->Fin_Year);
                    }

                    // Check if financial year exceeds 2 years
                    $start = Carbon::parse($startDate);
                    $end = Carbon::parse($endDate);

                    if ($start->diffInYears($end) > 2) {
                        $validator->errors()->add('Fin_Ed_Dt', 'Financial year cannot exceed 2 years.');
                    }
                }

                // Prevent status change if there are active dependencies
                if ($request->has('Fin_Status') && $request->Fin_Status != 'Active' && $finYear->Fin_Status == 'Active') {
                    // Check if this is the only active financial year
                    $activeYears = OrganisationFinYear::where('Org_Id', $finYear->Org_Id)
                        ->where('Fin_Status', 'Active')
                        ->where('Org_FYID', '!=', $id)
                        ->count();

                    if ($activeYears == 0) {
                        $validator->errors()->add('Fin_Status', 'Cannot deactivate the only active financial year for this organization. Please activate another year first.');
                    }
                }
            });

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], Response::HTTP_BAD_REQUEST);
            }

            // Update data
            $data = $validator->validated();

            // Update financial year
            $finYear->update($data);

            // If status is being set to active, deactivate other active years
            if (isset($data['Fin_Status']) && $data['Fin_Status'] === 'Active') {
                OrganisationFinYear::where('Org_Id', $finYear->Org_Id)
                    ->where('Org_FYID', '!=', $id)
                    ->where('Fin_Status', 'Active')
                    ->update(['Fin_Status' => 'Inactive']);
            }

            DB::commit();

            // Refresh model to get updated data
            $finYear->refresh();
            $finYear->load('organisation');

            return response()->json([
                'success' => true,
                'message' => 'Financial year updated successfully',
                'data' => $finYear
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to update financial year',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            // Validate ID
            if (!is_numeric($id) || $id <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid financial year ID'
                ], Response::HTTP_BAD_REQUEST);
            }

            $finYear = OrganisationFinYear::find($id);

            if (!$finYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year not found'
                ], Response::HTTP_NOT_FOUND);
            }

            // Check if financial year can be deleted
            if ($finYear->Fin_Status === 'Active') {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete an active financial year. Please deactivate it first.'
                ], Response::HTTP_CONFLICT);
            }

            // Add your business logic checks here for dependencies
            // Example: Check if there are transactions in this financial year
            /*
            $hasTransactions = Transaction::where('Fin_Year_ID', $id)->exists();
            if ($hasTransactions) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete financial year because it has associated transactions.'
                ], Response::HTTP_CONFLICT);
            }
            */

            // Soft delete
            $finYear->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Financial year deleted successfully'
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete financial year',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Restore a soft deleted resource.
     */
    public function restore(string $id)
    {
        DB::beginTransaction();

        try {
            // Validate ID
            if (!is_numeric($id) || $id <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid financial year ID'
                ], Response::HTTP_BAD_REQUEST);
            }

            $finYear = OrganisationFinYear::withTrashed()->find($id);

            if (!$finYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year not found'
                ], Response::HTTP_NOT_FOUND);
            }

            if (!$finYear->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year is not deleted'
                ], Response::HTTP_BAD_REQUEST);
            }

            // Restore
            $finYear->restore();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Financial year restored successfully',
                'data' => $finYear
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to restore financial year',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Get all trashed financial years.
     */
    public function trashed(Request $request)
    {
        try {
            $query = OrganisationFinYear::onlyTrashed();

            // Apply filters
            if ($request->has('org_id')) {
                $query->where('Org_Id', $request->org_id);
            }

            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('Fin_Year', 'like', "%{$search}%")
                        ->orWhereHas('organisation', function ($q) use ($search) {
                            $q->where('Org_Code', 'like', "%{$search}%")
                              ->orWhere('Org_Name', 'like', "%{$search}%");
                        });
                });
            }

            // Sorting
            $sortField = $request->get('sort_by', 'deleted_at');
            $sortDirection = $request->get('sort_dir', 'desc');

            $query->orderBy($sortField, $sortDirection);

            // Pagination
            $perPage = $request->get('per_page', 15);
            $perPage = min(max(1, $perPage), 100);

            $trashedYears = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Deleted financial years retrieved successfully',
                'data' => $trashedYears
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve deleted financial years',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Force delete a financial year.
     */
    public function forceDelete(string $id)
    {
        DB::beginTransaction();

        try {
            // Validate ID
            if (!is_numeric($id) || $id <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid financial year ID'
                ], Response::HTTP_BAD_REQUEST);
            }

            $finYear = OrganisationFinYear::withTrashed()->find($id);

            if (!$finYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'Financial year not found'
                ], Response::HTTP_NOT_FOUND);
            }

            // Force delete
            $finYear->forceDelete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Financial year permanently deleted successfully'
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to permanently delete financial year',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Get financial years by organization ID.
     */
    public function getByOrganization($orgId)
    {
        try {
            if (!is_numeric($orgId) || $orgId <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid organization ID'
                ], Response::HTTP_BAD_REQUEST);
            }

            // Check if organization exists
            $organization = OrganisationSetting::find($orgId);
            if (!$organization) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organization not found'
                ], Response::HTTP_NOT_FOUND);
            }

            $finYears = OrganisationFinYear::with('organisation')
                ->where('Org_Id', $orgId)
                ->orderBy('Fin_St_Dt', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'Financial years retrieved successfully',
                'data' => $finYears
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve financial years',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Get current active financial year for organization.
     */
    public function getCurrentForOrganization($orgId)
    {
        try {
            if (!is_numeric($orgId) || $orgId <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid organization ID'
                ], Response::HTTP_BAD_REQUEST);
            }

            // Check if organization exists
            $organization = OrganisationSetting::find($orgId);
            if (!$organization) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organization not found'
                ], Response::HTTP_NOT_FOUND);
            }

            $currentYear = OrganisationFinYear::with('organisation')
                ->where('Org_Id', $orgId)
                ->currentYear()
                ->first();

            if (!$currentYear) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active financial year found for current date'
                ], Response::HTTP_NOT_FOUND);
            }

            return response()->json([
                'success' => true,
                'message' => 'Current financial year retrieved successfully',
                'data' => $currentYear
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve current financial year',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
}
