<?php

namespace App\Http\Controllers;

use App\Models\PayBandMaster;
use App\Models\PayCommissionMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PayBandMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $payBands = PayBandMaster::with('commission')->get();
        return response()->json([
            'success' => true,
            'data' => $payBands
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'commission_id' => 'required|exists:pay_commission_masters,commission_id',
            'pay_band_code' => 'required|string|max:20',
            'min_pay' => 'required|numeric|min:0',
            'max_pay' => 'required|numeric|min:0|gt:min_pay'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $payBand = PayBandMaster::create($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Band created successfully',
            'data' => $payBand->load('commission')
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $payBand = PayBandMaster::with('commission')->find($id);

        if (!$payBand) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Band not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $payBand
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $payBand = PayBandMaster::find($id);

        if (!$payBand) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Band not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'commission_id' => 'exists:pay_commission_masters,commission_id',
            'pay_band_code' => 'string|max:20',
            'min_pay' => 'numeric|min:0',
            'max_pay' => 'numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $payBand->update($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Band updated successfully',
            'data' => $payBand->load('commission')
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $payBand = PayBandMaster::find($id);

        if (!$payBand) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Band not found'
            ], 404);
        }

        $payBand->delete();

        return response()->json([
            'success' => true,
            'message' => 'Pay Band deleted successfully'
        ]);
    }

    /**
     * Get pay bands by commission
     */
    public function getByCommission(string $commissionId)
    {
        $commission = PayCommissionMaster::find($commissionId);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Commission not found'
            ], 404);
        }

        $payBands = PayBandMaster::where('commission_id', $commissionId)->get();

        return response()->json([
            'success' => true,
            'data' => $payBands
        ]);
    }
}
