<?php

namespace App\Http\Controllers;

use App\Models\PayCommissionMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PayCommissionMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $commissions = PayCommissionMaster::with('structureType')->get();
        return response()->json([
            'success' => true,
            'data' => $commissions
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'commission_name' => 'required|string|max:50',
            'structure_type_id' => 'nullable|exists:pay_structure_type_masters,structure_type_id',

            'effective_from' => 'required|date',
            'effective_to' => 'nullable|date',
            'description' => 'nullable|string',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $commission = PayCommissionMaster::create($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Commission created successfully',
            'data' => $commission
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $commission = PayCommissionMaster::with('structureType')->find($id);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Commission not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $commission
        ]);
    }


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $commission = PayCommissionMaster::find($id);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Commission not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'commission_name' => 'string|max:50',
            'structure_type_id' => 'nullable|exists:pay_structure_type_masters,structure_type_id',

            'effective_from' => 'date',
            'effective_to' => 'nullable|date',
            'description' => 'nullable|string',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $commission->update($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Commission updated successfully',
            'data' => $commission
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $commission = PayCommissionMaster::find($id);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Commission not found'
            ], 404);
        }

        $commission->delete();

        return response()->json([
            'success' => true,
            'message' => 'Pay Commission deleted successfully'
        ]);
    }

    /**
     * Get commission details with related data
     */
    public function getCommissionDetails(string $id)
    {
        $commission = PayCommissionMaster::with(['payScales', 'payBands', 'gradePays', 'payMatrixLevels'])->find($id);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Commission not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $commission
        ]);
    }
}
