<?php

namespace App\Http\Controllers;

use App\Models\PayMatrixLevelMaster;
use App\Models\PayCommissionMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PayMatrixLevelMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $levels = PayMatrixLevelMaster::with('commission',  'payScale')->get();
        return response()->json([
            'success' => true,
            'data' => $levels
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'commission_id' => 'required|exists:pay_commission_masters,commission_id',
            'pay_scale_id'   => 'nullable|exists:pay_scale_masters,pay_scale_id',

            'level_code' => 'required|string|max:10',
            'starting_basic' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $level = PayMatrixLevelMaster::create($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Matrix Level created successfully',
            'data' => $level->load(['commission', 'payScale'])
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $level = PayMatrixLevelMaster::with(['commission',   'payScale', 'matrixCells'])->find($id);

        if (!$level) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Matrix Level not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $level
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $level = PayMatrixLevelMaster::find($id);

        if (!$level) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Matrix Level not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'commission_id' => 'exists:pay_commission_masters,commission_id',
            'pay_scale_id'   => 'nullable|exists:pay_scale_masters,pay_scale_id',

            'level_code' => 'string|max:10',
            'starting_basic' => 'numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $level->update($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Matrix Level updated successfully',
            'data' => $level->load(['commission', 'payScale'])
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $level = PayMatrixLevelMaster::find($id);

        if (!$level) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Matrix Level not found'
            ], 404);
        }

        $level->delete();

        return response()->json([
            'success' => true,
            'message' => 'Pay Matrix Level deleted successfully'
        ]);
    }

    /**
     * Get levels by commission
     */
    public function getByCommission(string $commissionId)
    {
        $commission = PayCommissionMaster::find($commissionId);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Commission not found'
            ], 404);
        }

        $levels = PayMatrixLevelMaster::with('matrixCells', 'payScale')
            ->where('commission_id', $commissionId)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $levels
        ]);
    }






    /**
     * Get levels by pay scale
     */
    public function getByPayScale(string $payScaleId)
    {
        $levels = PayMatrixLevelMaster::with(['commission', 'payScale'])
            ->where('pay_scale_id', $payScaleId)
            ->get();

        if ($levels->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'No Pay Matrix Levels found for this Pay Scale'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $levels
        ]);
    }
}
