<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\PermissionType;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class PermissionTypeController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'search' => 'nullable|string|max:100',
                'with_permissions' => 'nullable|boolean',
                'per_page' => 'nullable|integer|min:1|max:100',
                'sort_by' => 'nullable|string|in:name,created_at,updated_at',
                'sort_direction' => 'nullable|string|in:asc,desc'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();
            $query = PermissionType::query();

            // Search functionality
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('name', 'LIKE', "%{$search}%")
                      ->orWhere('description', 'LIKE', "%{$search}%");
                });
            }

            // Sorting
            $sortBy = $validated['sort_by'] ?? 'name';
            $sortDirection = $validated['sort_direction'] ?? 'asc';
            $query->orderBy($sortBy, $sortDirection);

            // Load relationships if requested
            if ($request->boolean('with_permissions')) {
                $query->with(['permissions' => function($query) {
                    $query->orderBy('name', 'asc');
                }]);
            }

            $perPage = $request->get('per_page', 15);
            $permissionTypes = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $permissionTypes,
                'message' => 'Permission types retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve permission types',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => [
                'required',
                'string',
                'max:100',
                'unique:permission_types,name',
                'regex:/^[a-zA-Z0-9\s\-_]+$/'
            ],
            'description' => 'nullable|string|max:500',
            'is_active' => 'nullable|boolean',
            'metadata' => 'nullable|array',
            'metadata.*.key' => 'required_with:metadata|string',
            'metadata.*.value' => 'required_with:metadata|string'
        ], [
            'name.required' => 'Permission type name is required',
            'name.unique' => 'This permission type name already exists',
            'name.regex' => 'Permission type name can only contain letters, numbers, spaces, hyphens and underscores',
            'name.max' => 'Permission type name cannot exceed 100 characters',
            'description.max' => 'Description cannot exceed 500 characters'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $validated = $validator->validated();

            $permissionType = PermissionType::create([
                'name' => trim($validated['name']),
                'description' => isset($validated['description']) ? trim($validated['description']) : null,
                'is_active' => $validated['is_active'] ?? true,
                'metadata' => $validated['metadata'] ?? null
            ]);

            return response()->json([
                'success' => true,
                'data' => $permissionType->load('permissions'),
                'message' => 'Permission type created successfully'
            ], 201);

        } catch (\Exception $e) {
            \Log::error('Failed to create permission type: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to create permission type',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    public function show($id): JsonResponse
    {
        try {
            $validator = Validator::make(['id' => $id], [
                'id' => 'required|integer|exists:permission_types,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $permissionType = PermissionType::with(['permissions' => function($query) {
                $query->orderBy('name', 'asc');
            }])->find($id);

            return response()->json([
                'success' => true,
                'data' => $permissionType,
                'message' => 'Permission type retrieved successfully'
            ]);

        } catch (\Exception $e) {
            \Log::error('Failed to retrieve permission type: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve permission type',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        $validator = Validator::make(array_merge($request->all(), ['id' => $id]), [
            'id' => 'required|integer|exists:permission_types,id',
            'name' => [
                'sometimes',
                'string',
                'max:100',
                Rule::unique('permission_types', 'name')->ignore($id),
                'regex:/^[a-zA-Z0-9\s\-_]+$/'
            ],
            'description' => 'nullable|string|max:500',
            'is_active' => 'sometimes|boolean',
            'metadata' => 'nullable|array',
            'metadata.*.key' => 'required_with:metadata|string',
            'metadata.*.value' => 'required_with:metadata|string'
        ], [
            'name.unique' => 'This permission type name already exists',
            'name.regex' => 'Permission type name can only contain letters, numbers, spaces, hyphens and underscores',
            'name.max' => 'Permission type name cannot exceed 100 characters',
            'description.max' => 'Description cannot exceed 500 characters'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $validated = $validator->validated();
            $permissionType = PermissionType::find($id);

            // Prepare update data
            $updateData = [];
            if (isset($validated['name'])) {
                $updateData['name'] = trim($validated['name']);
            }
            if (isset($validated['description'])) {
                $updateData['description'] = trim($validated['description']);
            }
            if (isset($validated['is_active'])) {
                $updateData['is_active'] = $validated['is_active'];
            }
            if (isset($validated['metadata'])) {
                $updateData['metadata'] = $validated['metadata'];
            }

            $permissionType->update($updateData);

            return response()->json([
                'success' => true,
                'data' => $permissionType->fresh(['permissions']),
                'message' => 'Permission type updated successfully'
            ]);

        } catch (\Exception $e) {
            \Log::error('Failed to update permission type: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to update permission type',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            $validator = Validator::make(['id' => $id], [
                'id' => 'required|integer|exists:permission_types,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $permissionType = PermissionType::withCount('permissions')->find($id);

            // Check if permission type has permissions assigned
            if ($permissionType->permissions_count > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete permission type. It has associated permissions.',
                    'data' => [
                        'permissions_count' => $permissionType->permissions_count
                    ]
                ], 409); // 409 Conflict is more appropriate
            }

            // Check if it's a system-defined permission type (optional)
            if ($permissionType->is_system ?? false) {
                return response()->json([
                    'success' => false,
                    'message' => 'System-defined permission types cannot be deleted'
                ], 403);
            }

            $permissionType->delete();

            return response()->json([
                'success' => true,
                'message' => 'Permission type deleted successfully'
            ]);

        } catch (\Exception $e) {
            \Log::error('Failed to delete permission type: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete permission type',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}
