<?php

// app/Http/Controllers/Api/RolePermissionController.php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\RolePermission;
use Illuminate\Http\Request;

class RolePermissionController extends Controller
{
    // 🔹 LIST ALL
    public function index()
    {
        return response()->json([
            'status' => true,
            'data' => RolePermission::with(['role', 'functionType'])->get()
        ]);
    }

    // 🔹 CREATE
    public function store(Request $request)
    {
        $request->validate([
            'role_id' => 'nullable|exists:role_master,Role_Id',
            'func_id' => 'nullable|exists:function_types,id',
            'actions' => 'nullable|array',
        ]);

        $permission = RolePermission::create([
            'role_id' => $request->role_id,
            'func_id' => $request->func_id,
            'actions' => $request->actions,
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Permission created successfully',
            'data' => $permission
        ], 201);
    }

    // 🔹 SHOW SINGLE
    public function show($id)
    {
        $permission = RolePermission::with(['role', 'functionType'])->findOrFail($id);

        return response()->json([
            'status' => true,
            'data' => $permission
        ]);
    }

    // 🔹 UPDATE (sometimes)
    public function update(Request $request, $id)
    {
        $permission = RolePermission::findOrFail($id);

        $request->validate([
            'role_id' => 'sometimes|nullable|exists:role_master,Role_Id',
            'func_id' => 'sometimes|nullable|exists:function_types,id',
            'actions' => 'sometimes|array',
        ]);

        // 🔥 MERGE ACTIONS
        if ($request->has('actions')) {
            $existingActions = $permission->actions ?? [];

            $mergedActions = array_merge(
                $existingActions,
                $request->actions
            );

            $permission->actions = $mergedActions;
        }

        // Other fields
        if ($request->has('role_id')) {
            $permission->role_id = $request->role_id;
        }

        if ($request->has('func_id')) {
            $permission->func_id = $request->func_id;
        }

        $permission->save();

        return response()->json([
            'status' => true,
            'message' => 'Permission updated successfully',
            'data' => $permission
        ]);
    }


    // 🔹 DELETE
    public function destroy($id)
    {
        RolePermission::findOrFail($id)->delete();

        return response()->json([
            'status' => true,
            'message' => 'Permission deleted'
        ]);
    }

    // 🔹 GET BY ROLE
    public function byRole($roleId)
    {
        $data = RolePermission::with('functionType', 'role')
            ->where('role_id', $roleId)
            ->get();

        return response()->json([
            'status' => true,
            'data' => $data
        ]);
    }
}
