<?php

namespace App\Http\Controllers;

use App\Models\SchemeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class SchemeMasterController extends Controller
{
    // 🔹 INDEX
    public function index()
    {
        try {
            $data = SchemeMaster::with('department')
                ->orderBy('scheme_id', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'Schemes fetched successfully',
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            // Map Dept_Id to department_id
            $data = $request->all();

            // If request has Dept_Id, map it to department_id
            if ($request->has('Dept_Id')) {
                $data['department_id'] = $request->input('Dept_Id');
            }

            $validator = Validator::make($data, [
                'scheme_code'   => 'nullable|string|max:30|unique:scheme_master,scheme_code',
                'scheme_name'   => 'required|string|max:200',
                'department_id' => 'nullable|exists:main_departments,id', // Fixed: now references main_departments
                'start_date'    => 'nullable|date',
                'end_date'      => 'nullable|date|after_or_equal:start_date',
                'budget'        => 'nullable|numeric|min:0',
                'is_active'     => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $scheme = SchemeMaster::create([
                'scheme_code' => $data['scheme_code'],
                'scheme_name' => $data['scheme_name'],
                'department_id' => $data['department_id'] ?? null,
                'start_date' => $data['start_date'] ?? null,
                'end_date' => $data['end_date'] ?? null,
                'budget' => $data['budget'] ?? 0,
                'status' => $data['status'] ?? 'ACTIVE',
                'is_active' => $data['is_active'] ?? true,
            ]);

            // Load department relation
            $scheme->load('department');

            return response()->json([
                'success' => true,
                'message' => 'Scheme created successfully',
                'data' => $scheme
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create scheme',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $scheme = SchemeMaster::with('department')->find($id);

            if (!$scheme) {
                return response()->json([
                    'success' => false,
                    'message' => 'Scheme not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $scheme
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching scheme',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE
   public function update(Request $request, $id)
{
    try {
        $scheme = SchemeMaster::find($id);

        if (!$scheme) {
            return response()->json([
                'success' => false,
                'message' => 'Scheme not found'
            ], 404);
        }

        // Map Dept_Id to department_id for validation
        $data = $request->all();

        // If request has Dept_Id, map it to department_id
        if ($request->has('Dept_Id')) {
            $data['department_id'] = $request->input('Dept_Id');
        }

        // Also handle direct department_id input
        if ($request->has('department_id')) {
            $data['department_id'] = $request->input('department_id');
        }

        $validator = Validator::make($data, [
            'scheme_code'   => 'sometimes|required|string|max:30|unique:scheme_master,scheme_code,' . $id . ',scheme_id',
            'scheme_name'   => 'sometimes|required|string|max:200',
            'department_id' => 'sometimes|nullable|exists:main_departments,id',
            'start_date'    => 'sometimes|nullable|date',
            'end_date'      => 'sometimes|nullable|date|after_or_equal:start_date',
            'budget'        => 'sometimes|nullable|numeric|min:0',
            'is_active'     => 'sometimes|boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        // Prepare update data
        $updateData = [];

        if ($request->has('scheme_code')) {
            $updateData['scheme_code'] = $request->input('scheme_code');
        }
        if ($request->has('scheme_name')) {
            $updateData['scheme_name'] = $request->input('scheme_name');
        }
        // Handle both Dept_Id and department_id
        if ($request->has('Dept_Id')) {
            $updateData['department_id'] = $request->input('Dept_Id');
        } elseif ($request->has('department_id')) {
            $updateData['department_id'] = $request->input('department_id');
        }
        if ($request->has('start_date')) {
            $updateData['start_date'] = $request->input('start_date');
        }
        if ($request->has('end_date')) {
            $updateData['end_date'] = $request->input('end_date');
        }
        if ($request->has('budget')) {
            $updateData['budget'] = $request->input('budget');
        }
        if ($request->has('status')) {
            $updateData['status'] = $request->input('status');
        }
        if ($request->has('is_active')) {
            $updateData['is_active'] = $request->input('is_active');
        }

        $scheme->update($updateData);

        // Reload with department
        $scheme->load('department');

        return response()->json([
            'success' => true,
            'message' => 'Scheme updated successfully',
            'data' => $scheme
        ], 200);
    } catch (Throwable $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to update scheme',
            'error' => $e->getMessage()
        ], 500);
    }
}

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $scheme = SchemeMaster::find($id);

            if (!$scheme) {
                return response()->json([
                    'success' => false,
                    'message' => 'Scheme not found'
                ], 404);
            }

            $scheme->delete();

            return response()->json([
                'success' => true,
                'message' => 'Scheme deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete scheme',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
