<?php

namespace App\Http\Controllers;

use App\Models\SignificantWorkMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class SignificantWorkMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = SignificantWorkMaster::where('is_active', true)->get();

            return response()->json([
                'success' => true,
                'message' => 'Significant works fetched successfully',
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'work_code'      => 'nullable|string|max:30|unique:significant_work_master,work_code',
                'work_name'      => 'required|string|max:200',
                'work_category'  => 'nullable|string|max:30',
                'estimated_cost' => 'nullable|numeric|min:0',
                'status'         => 'nullable|string|max:20'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $work = SignificantWorkMaster::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Significant work created successfully',
                'data' => $work
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create work',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $work = SignificantWorkMaster::find($id);

            if (!$work) {
                return response()->json([
                    'success' => false,
                    'message' => 'Work not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $work
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching work',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE (sometimes)
    public function update(Request $request, $id)
    {
        try {
            $work = SignificantWorkMaster::find($id);

            if (!$work) {
                return response()->json([
                    'success' => false,
                    'message' => 'Work not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'work_code'      => 'sometimes|required|string|max:30|unique:significant_work_master,work_code,' . $id . ',work_id',
                'work_name'      => 'sometimes|required|string|max:200',
                'work_category'  => 'sometimes|nullable|string|max:30',
                'estimated_cost' => 'sometimes|nullable|numeric|min:0',
                'status'         => 'sometimes|string|max:20',
                'is_active'      => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $work->update($request->only([
                'work_code',
                'work_name',
                'work_category',
                'estimated_cost',
                'status',
                'is_active'
            ]));

            return response()->json([
                'success' => true,
                'message' => 'Significant work updated successfully',
                'data' => $work
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update work',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE (Soft)
    public function destroy($id)
    {
        try {
            $work = SignificantWorkMaster::find($id);

            if (!$work) {
                return response()->json([
                    'success' => false,
                    'message' => 'Work not found'
                ], 404);
            }

            $work->update(['is_active' => false]);

            return response()->json([
                'success' => true,
                'message' => 'Work deactivated successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete work',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
