<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\SubSubCategory;
use App\Models\SubCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class SubSubCategoryController extends Controller
{
    /**
     * Display a listing of sub-subcategories (without pagination)
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        try {
            $subSubCategories = SubSubCategory::with(['subCategory.category'])
                ->orderBy('sub_sub_category_name')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - SubSubCategories fetched successfully',
                'data' => $subSubCategories
            ], 200);

        } catch (\Exception $e) {
            Log::error('SubSubCategory Fetch Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch sub-sub-categories. Please try again later.'
            ], 500);
        }
    }

    /**
     * Display paginated sub-subcategories (separate pagination API)
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function paginate(Request $request)
    {
        try {
            // Validate pagination parameters
            $validator = Validator::make($request->all(), [
                'per_page' => 'nullable|integer|min:1|max:100',
                'page' => 'nullable|integer|min:1',
                'search' => 'nullable|string|max:100',
                'sub_category_id' => 'nullable|integer|exists:sub_categories,sub_category_id',
                'category_id' => 'nullable|integer|exists:categories,category_id',
                'is_active' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid pagination parameters',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Get pagination parameters with defaults
            $perPage = $request->get('per_page', 10);
            $page = $request->get('page', 1);
            $search = $request->get('search', '');
            $subCategoryId = $request->get('sub_category_id', null);
            $categoryId = $request->get('category_id', null);
            $isActive = $request->get('is_active', null);

            // Build query
            $query = SubSubCategory::with(['subCategory.category'])
                ->when($search, function ($query, $search) {
                    return $query->where(function ($q) use ($search) {
                        $q->where('sub_sub_category_name', 'like', "%{$search}%")
                          ->orWhere('sub_sub_category_code', 'like', "%{$search}%")
                          ->orWhere('description', 'like', "%{$search}%");
                    });
                })
                ->when($subCategoryId, function ($query, $subCategoryId) {
                    return $query->where('sub_category_id', $subCategoryId);
                })
                ->when($categoryId, function ($query, $categoryId) {
                    return $query->whereHas('subCategory', function ($q) use ($categoryId) {
                        $q->where('category_id', $categoryId);
                    });
                })
                ->when(!is_null($isActive), function ($query) use ($isActive) {
                    return $query->where('is_active', $isActive);
                })
                ->orderBy('sub_sub_category_name');

            // Get paginated results
            $subSubCategories = $query->paginate($perPage, ['*'], 'page', $page);

            // Format pagination response
            $response = [
                'success' => true,
                'message' => 'GMC Gadhinagar - SubSubCategories paginated successfully',
                'data' => $subSubCategories->items(),
                'pagination' => [
                    'current_page' => $subSubCategories->currentPage(),
                    'first_page_url' => $subSubCategories->url(1),
                    'from' => $subSubCategories->firstItem(),
                    'last_page' => $subSubCategories->lastPage(),
                    'last_page_url' => $subSubCategories->url($subSubCategories->lastPage()),
                    'links' => $subSubCategories->linkCollection()->toArray(),
                    'next_page_url' => $subSubCategories->nextPageUrl(),
                    'path' => $subSubCategories->path(),
                    'per_page' => $subSubCategories->perPage(),
                    'prev_page_url' => $subSubCategories->previousPageUrl(),
                    'to' => $subSubCategories->lastItem(),
                    'total' => $subSubCategories->total(),
                ],
                'meta' => [
                    'total_records' => $subSubCategories->total(),
                    'current_page_records' => count($subSubCategories->items()),
                    'total_pages' => $subSubCategories->lastPage(),
                    'has_more_pages' => $subSubCategories->hasMorePages()
                ]
            ];

            return response()->json($response, 200);

        } catch (\Exception $e) {
            Log::error('SubSubCategory Paginate Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch paginated sub-sub-categories. Please try again later.'
            ], 500);
        }
    }

    /**
     * Store a newly created sub-subcategory.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'sub_category_id' => 'required|integer|exists:sub_categories,sub_category_id',
                'sub_sub_category_name' => 'required|string|max:100',
                'sub_sub_category_code' => 'required|string|max:50|regex:/^[A-Z0-9]+$/',
                'description' => 'nullable|string|max:500',
                'is_active' => 'boolean'
            ], [
                'sub_category_id.required' => 'Sub-category selection is required',
                'sub_category_id.exists' => 'Selected sub-category does not exist',
                'sub_sub_category_name.required' => 'Sub-sub-category name is required',
                'sub_sub_category_code.required' => 'Sub-sub-category code is required',
                'sub_sub_category_code.regex' => 'Sub-sub-category code should contain only uppercase letters and numbers'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            // Check unique constraint
            $exists = SubSubCategory::where('sub_category_id', $request->sub_category_id)
                ->where('sub_sub_category_code', strtoupper($request->sub_sub_category_code))
                ->exists();

            if ($exists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => [
                        'sub_sub_category_code' => ['This sub-sub-category code already exists for the selected sub-category']
                    ]
                ], 422);
            }

            $subSubCategory = SubSubCategory::create([
                'sub_category_id' => $request->sub_category_id,
                'sub_sub_category_name' => $request->sub_sub_category_name,
                'sub_sub_category_code' => strtoupper($request->sub_sub_category_code),
                'description' => $request->description,
                'is_active' => $request->boolean('is_active', true)
            ]);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - SubSubCategory created successfully',
                'data' => $subSubCategory->load(['subCategory.category'])
            ], 201);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            Log::error('SubSubCategory Create Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to create sub-sub-category. Please check the data and try again.'
            ], 500);
        }
    }

    /**
     * Display the specified sub-subcategory.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $subSubCategory = SubSubCategory::with(['subCategory.category'])->find($id);

            if (!$subSubCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-sub-category not found. Please check the sub-sub-category ID.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - SubSubCategory details retrieved successfully',
                'data' => $subSubCategory
            ], 200);

        } catch (\Exception $e) {
            Log::error('SubSubCategory Show Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve sub-sub-category details.'
            ], 500);
        }
    }

    /**
     * Update the specified sub-subcategory.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $subSubCategory = SubSubCategory::find($id);

            if (!$subSubCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-sub-category not found. Please check the sub-sub-category ID.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'sub_category_id' => 'required|integer|exists:sub_categories,sub_category_id',
                'sub_sub_category_name' => 'required|string|max:100',
                'sub_sub_category_code' => 'required|string|max:50|regex:/^[A-Z0-9]+$/',
                'description' => 'nullable|string|max:500',
                'is_active' => 'boolean'
            ], [
                'sub_category_id.required' => 'Sub-category selection is required',
                'sub_category_id.exists' => 'Selected sub-category does not exist',
                'sub_sub_category_name.required' => 'Sub-sub-category name is required',
                'sub_sub_category_code.required' => 'Sub-sub-category code is required',
                'sub_sub_category_code.regex' => 'Sub-sub-category code should contain only uppercase letters and numbers'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            // Check unique constraint (excluding current record)
            $exists = SubSubCategory::where('sub_category_id', $request->sub_category_id)
                ->where('sub_sub_category_code', strtoupper($request->sub_sub_category_code))
                ->where('sub_sub_category_id', '!=', $id)
                ->exists();

            if ($exists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => [
                        'sub_sub_category_code' => ['This sub-sub-category code already exists for the selected sub-category']
                    ]
                ], 422);
            }

            $subSubCategory->update([
                'sub_category_id' => $request->sub_category_id,
                'sub_sub_category_name' => $request->sub_sub_category_name,
                'sub_sub_category_code' => strtoupper($request->sub_sub_category_code),
                'description' => $request->description,
                'is_active' => $request->boolean('is_active', $subSubCategory->is_active)
            ]);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - SubSubCategory updated successfully',
                'data' => $subSubCategory->refresh()->load(['subCategory.category'])
            ], 200);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            Log::error('SubSubCategory Update Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to update sub-sub-category. Please check the data and try again.'
            ], 500);
        }
    }

    /**
     * Remove the specified sub-subcategory.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $subSubCategory = SubSubCategory::find($id);

            if (!$subSubCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-sub-category not found. Please check the sub-sub-category ID.'
                ], 404);
            }

            $subSubCategory->delete();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - SubSubCategory deleted successfully',
                'data' => null
            ], 200);

        } catch (\Illuminate\Database\QueryException $e) {
            Log::error('SubSubCategory Delete Query Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Cannot delete sub-sub-category due to database constraints.'
            ], 409);

        } catch (\Exception $e) {
            Log::error('SubSubCategory Delete Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete sub-sub-category. Please try again later.'
            ], 500);
        }
    }
}
