<?php

namespace App\Http\Controllers;

use App\Models\TaxTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class TaxTypeMasterController extends Controller
{
    // 🔹 INDEX
    public function index()
    {
        try {
            $data = TaxTypeMaster::orderBy('tax_type_id', 'desc')->get();

            return response()->json([
                'success' => true,
                'message' => 'Tax types fetched successfully',
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch tax types',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'tax_code' => 'nullable|string|max:20|unique:tax_type_master,tax_code',
                'tax_name' => 'required|string|max:100',
                'tax_percentage' => 'nullable|numeric|min:0',
                'frequency' => 'nullable|in:MONTHLY,QUARTERLY,HALF_YEARLY,YEARLY',
                'calculation_method' => 'nullable|string|max:30',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $tax = TaxTypeMaster::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Tax type created successfully',
                'data' => $tax
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create tax type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $tax = TaxTypeMaster::find($id);
            if (!$tax) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tax type not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $tax
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch tax type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE (sometimes)
    public function update(Request $request, $id)
    {
        try {
            $tax = TaxTypeMaster::find($id);
            if (!$tax) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tax type not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'tax_code' => 'sometimes|required|string|max:20|unique:tax_type_master,tax_code,' . $id . ',tax_type_id',
                'tax_name' => 'sometimes|required|string|max:100',
                'tax_percentage' => 'sometimes|nullable|numeric|min:0',
                'frequency' => 'sometimes|nullable|in:MONTHLY,QUARTERLY,HALF_YEARLY,YEARLY',
                'calculation_method' => 'sometimes|nullable|string|max:30',
                'is_active' => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $tax->update($request->only([
                'tax_code',
                'tax_name',
                'tax_percentage',
                'frequency',
                'calculation_method',
                'is_active'
            ]));

            return response()->json([
                'success' => true,
                'message' => 'Tax type updated successfully',
                'data' => $tax
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update tax type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $tax = TaxTypeMaster::find($id);
            if (!$tax) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tax type not found'
                ], 404);
            }

            $tax->delete();

            return response()->json([
                'success' => true,
                'message' => 'Tax type deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete tax type',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
