<?php

namespace App\Http\Controllers;

use App\Models\TendorAllocation;
use App\Models\TenderEntryMaster;
use App\Models\DocumentLibrary;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class TendorAllocationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = TendorAllocation::with([
                'tenderEntry',
                'tenderEntry.department',
                'tenderEntry.tenderWork',
                'subLedger',
                'subLedger.ledger',
                'documents'
            ]);

            // Search functionality
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('Gross_Amount', 'LIKE', "%{$search}%")
                        ->orWhereHas('tenderEntry', function ($q2) use ($search) {
                            $q2->where('Description', 'LIKE', "%{$search}%")
                                ->orWhere('Proposal_Amount', 'LIKE', "%{$search}%");
                        })
                        ->orWhereHas('subLedger', function ($q3) use ($search) {
                            $q3->where('name', 'LIKE', "%{$search}%")
                                ->orWhere('code', 'LIKE', "%{$search}%");
                        });
                });
            }

            // Filter by tender entry
            if ($request->has('tend_ent_id')) {
                $query->where('Tend_Ent_Id', $request->tend_ent_id);
            }

            // Filter by subledger
            if ($request->has('sl_id')) {
                $query->where('SL_Id', $request->sl_id);
            }

            // Order by
            $query->orderBy('created_at', 'desc');

            $perPage = $request->get('per_page', 15);
            $allocations = $query->paginate($perPage);

            // Format response
            $formattedAllocations = $allocations->through(function ($allocation) {
                $allocationData = $allocation->toArray();

                // Format uploaded documents
                $uploadedDocuments = [];
                if ($allocation->documents && $allocation->documents->count() > 0) {
                    foreach ($allocation->documents as $document) {
                        $docData = $document->toArray();

                        if (is_string($docData['Doc_Upload'])) {
                            $docData['Doc_Upload'] = json_decode($docData['Doc_Upload'], true);
                        }

                        $uploadedDocuments[] = $docData;
                    }
                }

                return [
                    'allocation' => $allocationData,
                    'uploaded_documents' => $uploadedDocuments
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $formattedAllocations,
                'message' => 'Tendor allocations retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving tendor allocations.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all tendor allocations without pagination
     */
    public function getAll(Request $request)
    {
        try {
            $query = TendorAllocation::with([
                'tenderEntry',
                'tenderEntry.department',
                'tenderEntry.tenderWork',
                'tenderEntry.ledger', // Added ledger for tender entry
                'subLedger',
                'subLedger.ledger',
                'documents'
            ]);

            // Search functionality (same as index)
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('Gross_Amount', 'LIKE', "%{$search}%")
                        ->orWhereHas('tenderEntry', function ($q2) use ($search) {
                            $q2->where('Description', 'LIKE', "%{$search}%")
                                ->orWhere('Proposal_Amount', 'LIKE', "%{$search}%");
                        })
                        ->orWhereHas('subLedger', function ($q3) use ($search) {
                            // Fix: Use correct column names from SubLedgerMaster model
                            $q3->where('SL_Name', 'LIKE', "%{$search}%")
                                ->orWhere('SL_Code', 'LIKE', "%{$search}%");
                        });
                });
            }

            // Filter by tender entry
            if ($request->has('tend_ent_id')) {
                $query->where('Tend_Ent_Id', $request->tend_ent_id);
            }

            // Filter by subledger
            if ($request->has('sl_id')) {
                $query->where('SL_Id', $request->sl_id);
            }

            // Filter by date range
            if ($request->has('start_date') && $request->has('end_date')) {
                $query->whereBetween('created_at', [$request->start_date, $request->end_date]);
            }

            // Filter by gross amount range
            if ($request->has('min_amount')) {
                $query->where('Gross_Amount', '>=', $request->min_amount);
            }

            if ($request->has('max_amount')) {
                $query->where('Gross_Amount', '<=', $request->max_amount);
            }

            // Order by
            $query->orderBy('created_at', 'desc');

            // Get all without pagination
            $allocations = $query->get();

            // Format response (same structure as index but without pagination)
            $formattedAllocations = $allocations->map(function ($allocation) {
                $allocationData = $allocation->toArray();

                // Ensure Doc_Upload is decoded for allocation documents
                if (isset($allocationData['documents']) && is_array($allocationData['documents'])) {
                    foreach ($allocationData['documents'] as &$doc) {
                        if (isset($doc['Doc_Upload']) && is_string($doc['Doc_Upload'])) {
                            $doc['Doc_Upload'] = json_decode($doc['Doc_Upload'], true);
                        }
                    }
                }

                // Format uploaded documents separately (like in index)
                $uploadedDocuments = [];
                if ($allocation->documents && $allocation->documents->count() > 0) {
                    foreach ($allocation->documents as $document) {
                        $docData = $document->toArray();

                        if (is_string($docData['Doc_Upload'])) {
                            $docData['Doc_Upload'] = json_decode($docData['Doc_Upload'], true);
                        }

                        $uploadedDocuments[] = $docData;
                    }
                }

                // Decode Multi_Doc if it's a string
                if (isset($allocationData['Multi_Doc']) && is_string($allocationData['Multi_Doc'])) {
                    $allocationData['Multi_Doc'] = json_decode($allocationData['Multi_Doc'], true);
                }

                return [
                    'allocation' => $allocationData,
                    'uploaded_documents' => $uploadedDocuments
                ];
            });

            // Get statistics
            $totalAllocations = $allocations->count();
            $totalGrossAmount = $allocations->sum('Gross_Amount');
            $averageAmount = $totalAllocations > 0 ? $totalGrossAmount / $totalAllocations : 0;

            return response()->json([
                'success' => true,
                'data' => [
                    'allocations' => $formattedAllocations,
                    'statistics' => [
                        'total_allocations' => $totalAllocations,
                        'total_gross_amount' => $totalGrossAmount,
                        'average_amount' => round($averageAmount, 2)
                    ]
                ],
                'message' => 'All tendor allocations retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            \Log::error('Get All Tendor Allocations Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving tendor allocations.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    /**
     * Store a newly created resource in storage.
     */
    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Tend_Ent_Id'  => 'required|exists:tender_entry_master,Tend_Ent_Id',
            'SL_Id'        => 'required|exists:subledg_master,SL_Id',
            'Gross_Amount' => 'required|numeric|min:0.01',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors'  => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();

        try {
            // 🔒 Lock tender row
            $tender = TenderEntryMaster::where('Tend_Ent_Id', $request->Tend_Ent_Id)
                ->lockForUpdate()
                ->firstOrFail();

            // ❌ Validation
            if ($request->Gross_Amount > $tender->Proposal_Amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gross Amount cannot exceed Proposal Amount'
                ], 422);
            }

            if ($request->Gross_Amount > $tender->available_amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient Available Amount'
                ], 422);
            }

            // ✅ Create allocation
            $allocation = TendorAllocation::create([
                'Tend_Ent_Id'  => $request->Tend_Ent_Id,
                'SL_Id'        => $request->SL_Id,
                'Gross_Amount' => $request->Gross_Amount,
            ]);

            // ✅ Deduct available amount
            $tender->available_amount -= $request->Gross_Amount;
            $tender->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Allocation created successfully',
                'data'    => [
                    'allocation' => $allocation,
                    'remaining_available_amount' => $tender->available_amount
                ]
            ], 201);

        } catch (Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $allocation = TendorAllocation::with([
                'tenderEntry',
                'tenderEntry.department',
                'tenderEntry.tenderWork',
                'tenderEntry.ledger',
                'subLedger',
                'subLedger.ledger',
                'documents'
            ])->findOrFail($id);

            // Format uploaded documents
            $uploadedDocuments = [];
            if ($allocation->documents && $allocation->documents->count() > 0) {
                foreach ($allocation->documents as $document) {
                    $docData = $document->toArray();

                    if (is_string($docData['Doc_Upload'])) {
                        $docData['Doc_Upload'] = json_decode($docData['Doc_Upload'], true);
                    }

                    $uploadedDocuments[] = $docData;
                }
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'allocation' => $allocation,
                    'uploaded_documents' => $uploadedDocuments
                ],
                'message' => 'Tendor allocation retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Tendor allocation not found.'
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'Gross_Amount' => 'required|numeric|min:0.01',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors'  => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();

        try {
            $allocation = TendorAllocation::findOrFail($id);

            $tender = TenderEntryMaster::where('Tend_Ent_Id', $allocation->Tend_Ent_Id)
                ->lockForUpdate()
                ->firstOrFail();

            $oldAmount = $allocation->Gross_Amount;
            $newAmount = $request->Gross_Amount;

            $difference = $newAmount - $oldAmount;

            // ❌ Proposal check
            if ($newAmount > $tender->Proposal_Amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gross Amount cannot exceed Proposal Amount'
                ], 422);
            }

            // ❌ Available check (only if increasing)
            if ($difference > 0 && $difference > $tender->available_amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient Available Amount'
                ], 422);
            }

            // ✅ Update allocation
            $allocation->Gross_Amount = $newAmount;
            $allocation->save();

            // ✅ Adjust available amount
            $tender->available_amount -= $difference;
            $tender->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Allocation updated successfully',
                'remaining_available_amount' => $tender->available_amount
            ], 200);

        } catch (Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $allocation = TendorAllocation::findOrFail($id);

            DB::beginTransaction();

            // Get and delete associated documents
            $documents = DocumentLibrary::where('Menu_Id', 2) // Assuming Menu_Id = 2 for Tendor Allocation
                ->where('Rec_Id', (string) $id)
                ->get();

            // Delete files from storage
            foreach ($documents as $document) {
                if (is_string($document->Doc_Upload)) {
                    $uploadData = json_decode($document->Doc_Upload, true);
                    if (isset($uploadData['public_url'])) {
                        $urlParts = parse_url($uploadData['public_url']);
                        if (isset($urlParts['path'])) {
                            $filename = basename($urlParts['path']);
                            $filePath = public_path('storage/tendor-allocation-documents/' . $filename);
                            if (file_exists($filePath)) {
                                unlink($filePath);
                            }
                        }
                    }
                }
            }

            // Delete document records
            DocumentLibrary::where('Menu_Id', 2)
                ->where('Rec_Id', (string) $id)
                ->delete();

            // Delete allocation
            $allocation->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Tendor allocation and associated documents deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error deleting tendor allocation.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get allocation statistics
     */
    public function statistics()
    {
        try {
            $totalAllocations = TendorAllocation::count();
            $totalGrossAmount = TendorAllocation::sum('Gross_Amount');
            $latestAllocation = TendorAllocation::with(['tenderEntry', 'subLedger'])
                ->orderBy('created_at', 'desc')
                ->first();

            // Count allocations by tender entry
            $allocationsByTender = TendorAllocation::select('Tend_Ent_Id', DB::raw('count(*) as count'))
                ->groupBy('Tend_Ent_Id')
                ->get();

            // Count documents for allocations
            $totalDocuments = DocumentLibrary::where('Menu_Id', 2) // Menu_Id = 2 for allocations
                ->where('Rec_Id', '!=', '')
                ->count();

            return response()->json([
                'success' => true,
                'data' => [
                    'total_allocations' => $totalAllocations,
                    'total_gross_amount' => $totalGrossAmount,
                    'latest_allocation' => $latestAllocation,
                    'allocations_by_tender' => $allocationsByTender,
                    'total_documents' => $totalDocuments,
                    'average_allocation_amount' => $totalAllocations > 0 ? $totalGrossAmount / $totalAllocations : 0
                ],
                'message' => 'Statistics retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving statistics.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get documents for a specific allocation
     */
    public function getDocuments(string $id)
    {
        try {
            $allocation = TendorAllocation::findOrFail($id);

            $documents = DocumentLibrary::where('Menu_Id', 2) // Menu_Id = 2 for allocations
                ->where('Rec_Id', (string) $id)
                ->orderBy('Serial_No', 'asc')
                ->get()
                ->map(function ($document) {
                    $docData = $document->toArray();

                    if (is_string($docData['Doc_Upload'])) {
                        $docData['Doc_Upload'] = json_decode($docData['Doc_Upload'], true);
                    }

                    return $docData;
                });

            return response()->json([
                'success' => true,
                'data' => $documents,
                'message' => 'Allocation documents retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving documents.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download allocation document
     */
    public function downloadDocument(string $docLibId)
    {
        try {
            $document = DocumentLibrary::findOrFail($docLibId);

            if (is_string($document->Doc_Upload)) {
                $uploadData = json_decode($document->Doc_Upload, true);

                if (isset($uploadData['public_url'], $uploadData['original_name'])) {
                    $urlParts = parse_url($uploadData['public_url']);
                    if (isset($urlParts['path'])) {
                        $filename = basename($urlParts['path']);
                        $filePath = public_path('storage/tendor-allocation-documents/' . $filename);

                        if (file_exists($filePath)) {
                            return response()->download($filePath, $uploadData['original_name']);
                        }
                    }
                }
            }

            return response()->json([
                'success' => false,
                'message' => 'File not found.'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error downloading document.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get allocations by tender entry
     */
    public function getByTenderEntry(string $tenderEntryId)
    {
        try {
            $allocations = TendorAllocation::with([
                'subLedger',
                'subLedger.ledger',
                'documents'
            ])->where('Tend_Ent_Id', $tenderEntryId)
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $allocations,
                'message' => 'Allocations retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving allocations.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
