<?php

namespace App\Http\Controllers;

use App\Models\WardStatistic;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class WardStatisticController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = WardStatistic::with(['ward', 'ward.zone', 'ward.zone.organisation'])
                        ->active();

            // Apply filters
            if ($request->has('ward_id')) {
                $query->byWard($request->ward_id);
            }

            if ($request->has('date')) {
                $query->byDate($request->date);
            }

            if ($request->has('start_date') && $request->has('end_date')) {
                $query->dateRange($request->start_date, $request->end_date);
            }

            if ($request->has('search')) {
                $query->search($request->search);
            }

            // Sorting
            $sortBy = $request->get('sort_by', 'St_Date');
            $sortOrder = $request->get('sort_order', 'desc');
            $query->orderBy($sortBy, $sortOrder);

            // Pagination
            $perPage = $request->get('per_page', 15);
            $statistics = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $statistics,
                'message' => 'Ward statistics retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve ward statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'Ward_Id' => 'required|exists:ward_master,Ward_Id',
                'St_Date' => 'required|string|max:25',
                'Population' => 'nullable|string|max:1',
                'Area_SqKm' => 'nullable|numeric|min:0|max:99999999.99',
            ], WardStatistic::validationMessages());

            // Custom validation for unique statistics per ward and date
            $validator->after(function ($validator) use ($request) {
                if (!WardStatistic::validateUniqueStatistic($request->Ward_Id, $request->St_Date)) {
                    $validator->errors()->add('St_Date', 'Statistics already exist for this ward on this date');
                }
            });

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $statistic = WardStatistic::create([
                'Ward_Id' => $request->Ward_Id,
                'St_Date' => $request->St_Date,
                'Population' => $request->Population,
                'Area_SqKm' => $request->Area_SqKm,
                'is_active' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Ward statistic created successfully',
                'data' => $statistic->load(['ward', 'ward.zone', 'ward.zone.organisation'])
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create ward statistic',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            $statistic = WardStatistic::with(['ward', 'ward.zone', 'ward.zone.organisation'])
                        ->find($id);

            if (!$statistic) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ward statistic not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $statistic,
                'message' => 'Ward statistic retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve ward statistic',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $statistic = WardStatistic::find($id);

            if (!$statistic) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ward statistic not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Ward_Id' => 'sometimes|required|exists:ward_master,Ward_Id',
                'St_Date' => 'sometimes|required|string|max:25',
                'Population' => 'nullable|string|max:1',
                'Area_SqKm' => 'nullable|numeric|min:0|max:99999999.99',
                'is_active' => 'sometimes|boolean'
            ], WardStatistic::validationMessages());

            // Custom validation for unique statistics per ward and date
            $validator->after(function ($validator) use ($request, $id) {
                if ($request->has('Ward_Id') && $request->has('St_Date')) {
                    if (!WardStatistic::validateUniqueStatistic($request->Ward_Id, $request->St_Date, $id)) {
                        $validator->errors()->add('St_Date', 'Statistics already exist for this ward on this date');
                    }
                }
            });

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $statistic->update($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Ward statistic updated successfully',
                'data' => $statistic->load(['ward', 'ward.zone', 'ward.zone.organisation'])
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update ward statistic',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $statistic = WardStatistic::find($id);

            if (!$statistic) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ward statistic not found'
                ], 404);
            }

            if (!$statistic->canBeDeleted()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete statistic as it has dependent records'
                ], 400);
            }

            $statistic->deactivate();

            return response()->json([
                'success' => true,
                'message' => 'Ward statistic deactivated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to deactivate ward statistic',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Restore deactivated ward statistic.
     */
    public function restore($id)
    {
        try {
            $statistic = WardStatistic::withTrashed()->find($id);

            if (!$statistic) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ward statistic not found'
                ], 404);
            }

            $statistic->activate();

            return response()->json([
                'success' => true,
                'message' => 'Ward statistic activated successfully',
                'data' => $statistic->load(['ward', 'ward.zone', 'ward.zone.organisation'])
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to activate ward statistic',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get statistics by ward.
     */
    public function getByWard($wardId)
    {
        try {
            $statistics = WardStatistic::with(['ward', 'ward.zone'])
                        ->byWard($wardId)
                        ->active()
                        ->latestFirst()
                        ->get();

            return response()->json([
                'success' => true,
                'data' => $statistics,
                'message' => 'Ward statistics retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve ward statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get latest statistics for each ward.
     */
    public function getLatestStatistics(Request $request)
    {
        try {
            $query = WardStatistic::with(['ward', 'ward.zone'])
                        ->active()
                        ->whereIn('WD_St_Id', function ($query) {
                            $query->selectRaw('MAX(WD_St_Id)')
                                  ->from('ward_statistics')
                                  ->whereColumn('Ward_Id', 'ward_statistics.Ward_Id')
                                  ->groupBy('Ward_Id');
                        });

            if ($request->has('zone_id')) {
                $query->whereHas('ward', function ($q) use ($request) {
                    $q->where('Zone_Id', $request->zone_id);
                });
            }

            $statistics = $query->get();

            return response()->json([
                'success' => true,
                'data' => $statistics,
                'message' => 'Latest ward statistics retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve latest statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk create ward statistics.
     */
    public function bulkStore(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'statistics' => 'required|array',
                'statistics.*.Ward_Id' => 'required|exists:ward_master,Ward_Id',
                'statistics.*.St_Date' => 'required|string|max:25',
                'statistics.*.Population' => 'nullable|string|max:1',
                'statistics.*.Area_SqKm' => 'nullable|numeric|min:0|max:99999999.99',
            ], WardStatistic::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $createdStatistics = [];
            $errors = [];

            foreach ($request->statistics as $index => $statData) {
                // Check for duplicate statistics
                if (!WardStatistic::validateUniqueStatistic($statData['Ward_Id'], $statData['St_Date'])) {
                    $errors[] = [
                        'index' => $index,
                        'message' => 'Statistics already exist for ward ' . $statData['Ward_Id'] . ' on date ' . $statData['St_Date']
                    ];
                    continue;
                }

                $statistic = WardStatistic::create([
                    'Ward_Id' => $statData['Ward_Id'],
                    'St_Date' => $statData['St_Date'],
                    'Population' => $statData['Population'] ?? null,
                    'Area_SqKm' => $statData['Area_SqKm'] ?? null,
                    'is_active' => true
                ]);

                $createdStatistics[] = $statistic;
            }

            return response()->json([
                'success' => true,
                'message' => count($createdStatistics) . ' ward statistics created successfully',
                'data' => $createdStatistics,
                'errors' => $errors,
                'total_created' => count($createdStatistics),
                'total_failed' => count($errors)
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create bulk ward statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
