<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class BudgetLogBook extends Model
{
    use HasFactory;

    // Table name
    protected $table = 'budgetlogbook';

    // Primary key
    protected $primaryKey = 'BL_Id';

    // Timestamps
    public $timestamps = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'Budg_Id',
        'Status',
        'previous_status',
        'Date',
        'user_id',
        'budget_amount',
        'updated_amount',
        'remarks'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'BL_Id' => 'integer',
        'Budg_Id' => 'integer',
        'user_id' => 'integer',
        'budget_amount' => 'decimal:2',
        'updated_amount' => 'decimal:2',
        'Date' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Relationship with BudgetEntry
     */
    public function budgetEntry()
    {
        return $this->belongsTo(BudgetEntry::class, 'Budg_Id', 'Budg_Id');
    }

    /**
     * Relationship with User
     */
    public function user()
    {
        return $this->belongsTo(\App\Models\GmcUser::class, 'user_id');
    }

    /**
     * Get formatted status
     */
    public function getFormattedStatusAttribute()
    {
        $statusMap = [
            'Draft' => 'Draft',
            'Revised' => 'Revised',
            'Approve' => 'Approved',
            'Freeze' => 'Frozen',
            'A' => 'Active',
            'I' => 'Inactive'
        ];

        return $statusMap[$this->Status] ?? $this->Status;
    }

    /**
     * Get formatted date
     */
    public function getFormattedDateAttribute()
    {
        return $this->Date ? $this->Date->format('Y-m-d H:i:s') : null;
    }

    /**
     * Get amount change (difference)
     */
    public function getAmountChangeAttribute()
    {
        return $this->updated_amount - $this->budget_amount;
    }

    /**
     * Get formatted amount change
     */
    public function getFormattedAmountChangeAttribute()
    {
        $change = $this->amount_change;
        if ($change == 0) {
            return 'No change';
        }

        $sign = $change > 0 ? '+' : '';
        return $sign . '₹ ' . number_format(abs($change), 2);
    }

    /**
     * Scope for filtering by budget entry ID
     */
    public function scopeByBudgetEntry($query, $budgId)
    {
        return $query->where('Budg_Id', $budgId);
    }

    /**
     * Scope for filtering by status
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('Status', $status);
    }

    /**
     * Scope for filtering by date range
     */
    public function scopeDateBetween($query, $startDate, $endDate)
    {
        return $query->whereBetween('Date', [$startDate, $endDate]);
    }

    /**
     * Scope for filtering by user
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }
}
