<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class CccMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'ccc_master';
    protected $primaryKey = 'CCC_Id';

    protected $fillable = [
        'CCC_Code',
        'CCC_Name',
        'Ward_Id',
        'is_active'
    ];

    protected $casts = [
        'CCC_Id' => 'integer',
        'Ward_Id' => 'integer',
        'is_active' => 'boolean'
    ];

    // Validation rules for create
    public static function createRules()
    {
        return [
            'CCC_Code' => 'required|string|max:5|unique:ccc_master,CCC_Code',
            'CCC_Name' => 'required|string|max:25',
            'Ward_Id' => 'required|exists:ward_master,Ward_Id'
        ];
    }

    // Validation rules for update
    public static function updateRules($id)
    {
        return [
            'CCC_Code' => 'sometimes|required|string|max:5|unique:ccc_master,CCC_Code,' . $id . ',CCC_Id',
            'CCC_Name' => 'sometimes|required|string|max:25',
            'Ward_Id' => 'sometimes|required|exists:ward_master,Ward_Id',
            'is_active' => 'sometimes|boolean'
        ];
    }

    // Custom validation messages
    public static function validationMessages()
    {
        return [
            'CCC_Code.required' => 'CCC code is required',
            'CCC_Code.max' => 'CCC code cannot exceed 5 characters',
            'CCC_Code.unique' => 'This CCC code already exists',
            'CCC_Name.required' => 'CCC name is required',
            'CCC_Name.max' => 'CCC name cannot exceed 25 characters',
            'Ward_Id.required' => 'Ward is required',
            'Ward_Id.exists' => 'The selected ward does not exist'
        ];
    }

    // Relationship with ward
    public function ward()
    {
        return $this->belongsTo(WardMaster::class, 'Ward_Id', 'Ward_Id');
    }

    // Relationship with zone through ward
    public function zone()
    {
        return $this->hasOneThrough(
            ZoneMaster::class,
            WardMaster::class,
            'Ward_Id', // Foreign key on ward_master table
            'Zone_Id', // Foreign key on zone_master table
            'Ward_Id', // Local key on ccc_master table
            'Zone_Id'  // Local key on ward_master table
        );
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByWard($query, $wardId)
    {
        return $query->where('Ward_Id', $wardId);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('CCC_Code', 'like', "%{$search}%")
              ->orWhere('CCC_Name', 'like', "%{$search}%")
              ->orWhereHas('ward', function ($q) use ($search) {
                  $q->where('Ward_Code', 'like', "%{$search}%")
                    ->orWhere('Ward_Name', 'like', "%{$search}%")
                    ->orWhereHas('zone', function ($q) use ($search) {
                        $q->where('Zone_Code', 'like', "%{$search}%")
                          ->orWhere('Zone_Name', 'like', "%{$search}%");
                    });
              });
        });
    }

    // Attribute accessors
    public function getWardNameAttribute()
    {
        return $this->ward ? $this->ward->Ward_Name : null;
    }

    public function getWardCodeAttribute()
    {
        return $this->ward ? $this->ward->Ward_Code : null;
    }

    public function getZoneNameAttribute()
    {
        return $this->ward && $this->ward->zone ? $this->ward->zone->Zone_Name : null;
    }

    public function getZoneCodeAttribute()
    {
        return $this->ward && $this->ward->zone ? $this->ward->zone->Zone_Code : null;
    }

    public function getFullCccAttribute()
    {
        return $this->CCC_Code . ' - ' . $this->CCC_Name;
    }

    // Business logic methods
    public function canBeDeleted()
    {
        // Check if CCC has any dependencies
        // Example: check if there are any transactions or records linked to this CCC
        return true;
    }

    public function deactivate()
    {
        $this->update(['is_active' => false]);
        return $this;
    }

    public function activate()
    {
        $this->update(['is_active' => true]);
        return $this;
    }
}
