<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CoaDepMapping extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'coa_dep_mapping';

    /**
     * The primary key for the model.
     *
     * @var string
     */
    protected $primaryKey = 'coa_dep_map';

    /**
     * Indicates if the model's ID is auto-incrementing.
     *
     * @var bool
     */
    public $incrementing = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'Dep_Id',
        'Gr_IDs', // Changed from Ledg_IDs to Gr_IDs
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'Gr_IDs' => 'array', // Cast JSON to array
    ];

    /**
     * Get the main department associated with this mapping.
     */
    public function mainDepartment()
    {
        return $this->belongsTo(MainDepartment::class, 'Dep_Id', 'id');
    }

    /**
     * Get all ledger groups associated with this mapping.
     */
    public function ledgerGroups()
    {
        if (!empty($this->Gr_IDs)) {
            return LedgGrp::whereIn('Gr_Id', $this->Gr_IDs)->get();
        }
        return collect();
    }

    /**
     * Get first ledger group for backward compatibility.
     */
    public function getFirstLedgerGroupAttribute()
    {
        if (!empty($this->Gr_IDs)) {
            return LedgGrp::find($this->Gr_IDs[0]);
        }
        return null;
    }

    /**
     * Add a ledger group ID to the mapping.
     */
    public function addGroupId($groupId)
    {
        $groupIds = $this->Gr_IDs ?? [];

        if (!in_array($groupId, $groupIds)) {
            $groupIds[] = $groupId;
            $this->Gr_IDs = $groupIds;
            return $this->save();
        }

        return false;
    }

    /**
     * Remove a ledger group ID from the mapping.
     */
    public function removeGroupId($groupId)
    {
        $groupIds = $this->Gr_IDs ?? [];
        $index = array_search($groupId, $groupIds);

        if ($index !== false) {
            unset($groupIds[$index]);
            $this->Gr_IDs = array_values($groupIds); // Re-index
            return $this->save();
        }

        return false;
    }

    /**
     * Check if ledger group ID exists in mapping.
     */
    public function hasGroupId($groupId)
    {
        $groupIds = $this->Gr_IDs ?? [];
        return in_array($groupId, $groupIds);
    }

    /**
     * Get all group IDs as array.
     */
    public function getGroupIdsAttribute()
    {
        return $this->Gr_IDs ?? [];
    }

    /**
     * Scopes
     */
    public function scopeByDepartment($query, $departmentId)
    {
        return $query->where('Dep_Id', $departmentId);
    }

    public function scopeByGroup($query, $groupId)
    {
        return $query->whereJsonContains('Gr_IDs', $groupId);
    }

    public function scopeActive($query)
    {
        return $query->whereHas('ledgerGroups', function ($q) {
            $q->where('is_del', '!=', 'Y');
        });
    }
}
