<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class FaTaxTypeMaster extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'fa_tax_type_master';

    /**
     * The primary key associated with the table.
     *
     * @var string
     */
    protected $primaryKey = 'tax_type_id';

    /**
     * Indicates if the model's ID is auto-incrementing.
     *
     * @var bool
     */
    public $incrementing = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'Lg_Id',
        'tax_code',
        'tax_name',
        'tax_percentage'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'tax_percentage' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Validation rules for creating tax type
     */
    public static function createRules()
    {
        return [
            'Lg_Id' => 'nullable|integer|exists:ledger_master,Lg_Id',
            'tax_code' => 'required|string|max:20|unique:fa_tax_type_master',
            'tax_name' => 'required|string|max:100',
            'tax_percentage' => 'required|numeric|min:0|max:100'
        ];
    }

    /**
     * Validation rules for updating tax type
     */
    public static function updateRules($id)
    {
        return [
            'Lg_Id' => 'nullable|integer|exists:ledger_master,Lg_Id',
            'tax_code' => 'required|string|max:20|unique:fa_tax_type_master,tax_code,' . $id . ',tax_type_id',
            'tax_name' => 'required|string|max:100',
            'tax_percentage' => 'required|numeric|min:0|max:100'
        ];
    }

    /**
     * Custom validation messages
     */
    public static function validationMessages()
    {
        return [
            'Lg_Id.exists' => 'Selected ledger does not exist',
            'tax_code.required' => 'Tax code is required',
            'tax_code.unique' => 'This tax code already exists',
            'tax_code.max' => 'Tax code cannot exceed 20 characters',
            'tax_name.required' => 'Tax name is required',
            'tax_name.max' => 'Tax name cannot exceed 100 characters',
            'tax_percentage.required' => 'Tax percentage is required',
            'tax_percentage.numeric' => 'Tax percentage must be a number',
            'tax_percentage.min' => 'Tax percentage cannot be less than 0',
            'tax_percentage.max' => 'Tax percentage cannot exceed 100'
        ];
    }

    /**
     * Relationship with LedgerMaster
     */
    public function ledger()
    {
        return $this->belongsTo(LedgerMaster::class, 'Lg_Id', 'Lg_Id');
    }

    /**
     * Scope to filter by ledger ID
     */
    public function scopeByLedger($query, $ledgerId)
    {
        return $query->where('Lg_Id', $ledgerId);
    }

    /**
     * Scope to get active tax types (with ledger or without)
     */
    public function scopeActive($query)
    {
        return $query->where(function($q) {
            $q->whereNotNull('tax_code')
              ->where('tax_percentage', '>=', 0);
        });
    }

    /**
     * Get formatted tax percentage
     */
    public function getFormattedTaxPercentageAttribute()
    {
        return number_format($this->tax_percentage, 2) . '%';
    }

    /**
     * Check if tax type is associated with a ledger
     */
    public function hasLedger()
    {
        return !is_null($this->Lg_Id);
    }

    /**
     * Get ledger name if associated
     */
    public function getLedgerNameAttribute()
    {
        return $this->ledger ? $this->ledger->Lg_Name : null;
    }

    /**
     * Get ledger code if associated
     */
    public function getLedgerCodeAttribute()
    {
        return $this->ledger ? $this->ledger->Lg_Code : null;
    }
}
