<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class GmcDeviceApproval extends Model
{
    use HasFactory;

    protected $table = 'gmc_device_approvals';
    protected $primaryKey = 'approval_id';
    public $timestamps = true;

    protected $fillable = [
        'user_id',
        'device_unique_id',
        'device_name',
        'device_type',
        'device_category',
        'mac_address',
        'hdd_serial',
        'motherboard_serial',
        'processor_id',
        'browser_fingerprint',
        'ip_address',
        'operating_system',
        'browser',
        'department_id',
        'location_id',
        'requested_at',
        'requested_by',
        'approved_at',
        'approved_by',
        'approval_type',
        'valid_from',
        'valid_until',
        'status',
        'is_active',
        'is_primary_device'
    ];

    protected $casts = [
        'requested_at' => 'datetime',
        'approved_at' => 'datetime',
        'valid_from' => 'datetime',
        'valid_until' => 'datetime',
        'is_active' => 'boolean',
        'is_primary_device' => 'boolean'
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(GmcUser::class, 'user_id', 'user_id');
    }

    public function requester()
    {
        return $this->belongsTo(GmcUser::class, 'requested_by', 'user_id');
    }

    public function approver()
    {
        return $this->belongsTo(GmcUser::class, 'approved_by', 'user_id');
    }

    public function department()
    {
        return $this->belongsTo(Department::class, 'department_id', 'department_id');
    }

    public function logs()
    {
        return $this->hasMany(GmcDeviceApprovalLog::class, 'approval_id', 'approval_id');
    }

    // Helper Methods
    public function isValid()
    {
        if (!$this->is_active || $this->status !== 'ACTIVE') {
            return false;
        }

        return Carbon::now()->between($this->valid_from, $this->valid_until);
    }

    public function isExpired()
    {
        return Carbon::now()->greaterThan($this->valid_until);
    }

    public function expiresInDays($days = 30)
    {
        return $this->valid_until->diffInDays(Carbon::now()) <= $days;
    }

    public function renew($validityDays = 365, $approvedBy = null)
    {
        $this->update([
            'valid_until' => now()->addDays($validityDays),
            'approved_by' => $approvedBy ?? $this->approved_by,
            'approved_at' => now(),
            'status' => 'ACTIVE'
        ]);
    }

    public function revoke($reason = null)
    {
        $this->update([
            'status' => 'REVOKED',
            'is_active' => false
        ]);
    }

    // Static Methods
    public static function createFromQueue(GmcDeviceRegistrationQueue $queue, $approvedBy)
    {
        $registrationData = $queue->registration_data;
        $isCitizen = ($registrationData['Role_Id'] ?? 0) == 4;

        // Set validity: 1 year for employees, 10 years for citizens
        $validityDays = $isCitizen ? 3650 : 365;

        return self::create([
            'user_id' => $queue->user_id,
            'device_unique_id' => $queue->generateDeviceUniqueId(),
            'device_name' => $queue->device_name,
            'device_type' => $queue->device_type,
            'device_category' => 'PRIMARY',
            'mac_address' => $queue->mac_address,
            'hdd_serial' => $queue->hdd_serial,
            'motherboard_serial' => $queue->motherboard_serial,
            'processor_id' => $queue->processor_id,
            'browser_fingerprint' => $queue->browser_fingerprint,
            'ip_address' => $queue->ip_address,
            'operating_system' => $queue->operating_system,
            'browser' => $queue->browser,
            'department_id' => $queue->department_id,
            'location_id' => $queue->location_id,
            'requested_at' => $queue->submitted_at,
            'requested_by' => $queue->user_id,
            'approved_at' => now(),
            'approved_by' => $approvedBy,
            'approval_type' => $isCitizen ? 'AUTO_CITIZEN' : 'MANUAL',
            'valid_from' => now(),
            'valid_until' => now()->addDays($validityDays),
            'status' => 'ACTIVE',
            'is_active' => true,
            'is_primary_device' => true
        ]);
    }
}
