<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class GmcDeviceRegistrationQueue extends Model
{
    use HasFactory;

    protected $table = 'gmc_device_registration_queue';
    protected $primaryKey = 'queue_id';
    public $timestamps = true;

    protected $fillable = [
        'user_id',
        'registration_data',
        'ip_address',
        'mac_address',
        'hdd_serial',
        'motherboard_serial',
        'processor_id',
        'browser_fingerprint',
        'operating_system',
        'browser',
        'device_name',
        'device_type',
        'department_id',
        'location_id',
        'status',
        'rejection_reason',
        'processed_at',
        'processed_by',
        'is_citizen',
        'submitted_at'
    ];

    protected $casts = [
        'registration_data' => 'array',
        'submitted_at' => 'datetime',
        'processed_at' => 'datetime',
        'is_citizen' => 'boolean'
    ];

    /**
     * Relationship with user who registered
     */
    public function user()
    {
        return $this->belongsTo(GmcUser::class, 'user_id', 'user_id');
    }

    /**
     * Relationship with user who processed the request
     */
    public function processor()
    {
        return $this->belongsTo(GmcUser::class, 'processed_by', 'user_id');
    }

    /**
     * Relationship with department
     */
    public function department()
    {
        return $this->belongsTo(Department::class, 'department_id', 'department_id');
    }

    /**
     * Relationship with approval logs
     */
    public function logs()
    {
        return $this->hasMany(GmcDeviceApprovalLog::class, 'queue_id', 'queue_id');
    }

    /**
     * Check if pending
     */
    public function isPending()
    {
        return $this->status === 'PENDING';
    }

    /**
     * Check if approved
     */
    public function isApproved()
    {
        return $this->status === 'APPROVED';
    }

    /**
     * Check if rejected
     */
    public function isRejected()
    {
        return $this->status === 'REJECTED';
    }

    /**
     * Approve the device registration
     */
    public function approve($processedBy, $remarks = null)
    {
        $this->update([
            'status' => 'APPROVED',
            'processed_at' => now(),
            'processed_by' => $processedBy
        ]);

        // Create approval record
        if (class_exists('App\Models\GmcDeviceApproval')) {
            \App\Models\GmcDeviceApproval::createFromQueue($this, $processedBy);
        }
    }

    /**
     * Reject the device registration
     */
    public function reject($processedBy, $reason)
    {
        $this->update([
            'status' => 'REJECTED',
            'rejection_reason' => $reason,
            'processed_at' => now(),
            'processed_by' => $processedBy
        ]);
    }

    /**
     * Generate device unique ID
     */
    public function generateDeviceUniqueId()
    {
        $components = [
            $this->mac_address,
            $this->hdd_serial,
            $this->browser_fingerprint,
            $this->user_id
        ];

        return md5(implode('|', array_filter($components)));
    }
}
