<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class LedgerDepMapping extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'ledger_dep_mapping';

    /**
     * The primary key for the model.
     *
     * @var string
     */
    protected $primaryKey = 'Led_d_map';

    /**
     * Indicates if the model's ID is auto-incrementing.
     *
     * @var bool
     */
    public $incrementing = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'Dep_Id',
        'Ledg_IDs', // Changed to Ledg_IDs (plural)
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'Ledg_IDs' => 'array', // Cast JSON to array
    ];

    /**
     * Get the main department associated with this mapping.
     */
    public function mainDepartment()
    {
        return $this->belongsTo(MainDepartment::class, 'Dep_Id', 'id');
    }

    /**
     * Get all ledgers associated with this mapping.
     */
    public function ledgers()
    {
        if (!empty($this->Ledg_IDs)) {
            return LedgerMaster::whereIn('Lg_Id', $this->Ledg_IDs)->get();
        }
        return collect();
    }

    /**
     * Get first ledger for backward compatibility.
     */
    public function getFirstLedgerAttribute()
    {
        if (!empty($this->Ledg_IDs)) {
            return LedgerMaster::find($this->Ledg_IDs[0]);
        }
        return null;
    }

    /**
     * Add a ledger ID to the mapping.
     */
    public function addLedgerId($ledgerId)
    {
        $ledgerIds = $this->Ledg_IDs ?? [];

        if (!in_array($ledgerId, $ledgerIds)) {
            $ledgerIds[] = $ledgerId;
            $this->Ledg_IDs = $ledgerIds;
            return $this->save();
        }

        return false;
    }

    /**
     * Remove a ledger ID from the mapping.
     */
    public function removeLedgerId($ledgerId)
    {
        $ledgerIds = $this->Ledg_IDs ?? [];
        $index = array_search($ledgerId, $ledgerIds);

        if ($index !== false) {
            unset($ledgerIds[$index]);
            $this->Ledg_IDs = array_values($ledgerIds); // Re-index
            return $this->save();
        }

        return false;
    }

    /**
     * Check if ledger ID exists in mapping.
     */
    public function hasLedgerId($ledgerId)
    {
        $ledgerIds = $this->Ledg_IDs ?? [];
        return in_array($ledgerId, $ledgerIds);
    }
}
