<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Validation\Rule;

class OrganisationFinYear extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'organisation_fin_years';
    protected $primaryKey = 'Org_FYID';

    protected $fillable = [
        'Org_Id', // Now bigint
        'Fin_Year',
        'Fin_St_Dt',
        'Fin_Ed_Dt',
        'Fin_Status'
    ];

    protected $casts = [
        'Org_FYID' => 'integer',
        'Org_Id' => 'integer', // Cast to integer
        'Fin_St_Dt' => 'date:Y-m-d',
        'Fin_Ed_Dt' => 'date:Y-m-d'
    ];

    // Validation rules for create
    public static function createRules()
    {
        return [
            'Org_Id' => 'required|integer|exists:organisation_settings,Org_Id',
            'Fin_Year' => [
                'required',
                'string',
                'max:25',
                Rule::unique('organisation_fin_years')->where(function ($query) {
                    return $query->where('Org_Id', request()->Org_Id);
                })
            ],
            'Fin_St_Dt' => 'required|date|date_format:Y-m-d',
            'Fin_Ed_Dt' => 'required|date|date_format:Y-m-d|after:Fin_St_Dt',
            'Fin_Status' => 'nullable|in:Active,Inactive,Closed'
        ];
    }

    // Validation rules for update
    public static function updateRules($id)
    {
        return [
            'Org_Id' => 'sometimes|required|integer|exists:organisation_settings,Org_Id',
            'Fin_Year' => [
                'sometimes',
                'required',
                'string',
                'max:25',
                Rule::unique('organisation_fin_years')
                    ->where(function ($query) use ($id) {
                        $orgId = request()->Org_Id ?? OrganisationFinYear::find($id)->Org_Id;
                        return $query->where('Org_Id', $orgId);
                    })
                    ->ignore($id, 'Org_FYID')
            ],
            'Fin_St_Dt' => 'sometimes|required|date|date_format:Y-m-d',
            'Fin_Ed_Dt' => 'sometimes|required|date|date_format:Y-m-d|after:Fin_St_Dt',
            'Fin_Status' => 'nullable|in:Active,Inactive,Closed'
        ];
    }

    // Custom validation messages
    public static function validationMessages()
    {
        return [
            'Org_Id.required' => 'Organization ID is required',
            'Org_Id.integer' => 'Organization ID must be an integer',
            'Org_Id.exists' => 'The selected organization does not exist.',
            'Fin_Year.unique' => 'This financial year already exists for the selected organization.',
            'Fin_Ed_Dt.after' => 'End date must be after start date.',
            'Fin_Status.in' => 'Status must be one of: Active, Inactive, or Closed.'
        ];
    }

    // Relationships - Updated to reference Org_Id (bigint)
    public function organisation()
    {
        return $this->belongsTo(OrganisationSetting::class, 'Org_Id', 'Org_Id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('Fin_Status', 'Active');
    }

    public function scopeForOrganisation($query, $orgId)
    {
        return $query->where('Org_Id', $orgId);
    }

    public function scopeCurrentYear($query)
    {
        $today = now()->format('Y-m-d');
        return $query->where('Fin_St_Dt', '<=', $today)
                     ->where('Fin_Ed_Dt', '>=', $today)
                     ->where('Fin_Status', 'Active');
    }

    // Business logic methods
    public function isActive()
    {
        return $this->Fin_Status === 'Active';
    }

    public function isDateInRange($date)
    {
        $checkDate = is_string($date) ? $date : $date->format('Y-m-d');
        return $checkDate >= $this->Fin_St_Dt && $checkDate <= $this->Fin_Ed_Dt;
    }

    public function getDurationInDays()
    {
        $start = \Carbon\Carbon::parse($this->Fin_St_Dt);
        $end = \Carbon\Carbon::parse($this->Fin_Ed_Dt);
        return $start->diffInDays($end);
    }

    public function getDurationInMonths()
    {
        $start = \Carbon\Carbon::parse($this->Fin_St_Dt);
        $end = \Carbon\Carbon::parse($this->Fin_Ed_Dt);
        return $start->diffInMonths($end);
    }

    // Get organization details through relationship
    public function getOrganizationNameAttribute()
    {
        return $this->organisation ? $this->organisation->Org_Name : null;
    }

    public function getOrganizationCodeAttribute()
    {
        return $this->organisation ? $this->organisation->Org_Code : null;
    }
}
