<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PayrollMaster extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'payroll_master';

    /**
     * The primary key associated with the table.
     *
     * @var string
     */
    protected $primaryKey = 'payroll_id';

    /**
     * Indicates if the model's ID is auto-incrementing.
     *
     * @var bool
     */
    public $incrementing = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'Lg_Id',
        'payroll_code',
        'payroll_name',
        'payroll_percentage'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'payroll_percentage' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Validation rules for creating payroll
     */
    public static function createRules()
    {
        return [
            'Lg_Id' => 'nullable|integer|exists:ledger_master,Lg_Id',
            'payroll_code' => 'required|string|max:20|unique:payroll_master',
            'payroll_name' => 'required|string|max:100',
            'payroll_percentage' => 'required|numeric|min:0|max:100'
        ];
    }

    /**
     * Validation rules for updating payroll
     */
    public static function updateRules($id)
    {
        return [
            'Lg_Id' => 'nullable|integer|exists:ledger_master,Lg_Id',
            'payroll_code' => 'required|string|max:20|unique:payroll_master,payroll_code,' . $id . ',payroll_id',
            'payroll_name' => 'required|string|max:100',
            'payroll_percentage' => 'required|numeric|min:0|max:100'
        ];
    }

    /**
     * Custom validation messages
     */
    public static function validationMessages()
    {
        return [
            'Lg_Id.exists' => 'Selected ledger does not exist',
            'payroll_code.required' => 'Payroll code is required',
            'payroll_code.unique' => 'This payroll code already exists',
            'payroll_code.max' => 'Payroll code cannot exceed 20 characters',
            'payroll_name.required' => 'Payroll name is required',
            'payroll_name.max' => 'Payroll name cannot exceed 100 characters',
            'payroll_percentage.required' => 'Payroll percentage is required',
            'payroll_percentage.numeric' => 'Payroll percentage must be a number',
            'payroll_percentage.min' => 'Payroll percentage cannot be less than 0',
            'payroll_percentage.max' => 'Payroll percentage cannot exceed 100'
        ];
    }

    /**
     * Relationship with LedgerMaster
     */
    public function ledger()
    {
        return $this->belongsTo(LedgerMaster::class, 'Lg_Id', 'Lg_Id');
    }

    /**
     * Scope to filter by ledger ID
     */
    public function scopeByLedger($query, $ledgerId)
    {
        return $query->where('Lg_Id', $ledgerId);
    }

    /**
     * Scope to get active payrolls (with ledger or without)
     */
    public function scopeActive($query)
    {
        return $query->where(function($q) {
            $q->whereNotNull('payroll_code')
              ->where('payroll_percentage', '>=', 0);
        });
    }

    /**
     * Get formatted payroll percentage
     */
    public function getFormattedPayrollPercentageAttribute()
    {
        return number_format($this->payroll_percentage, 2) . '%';
    }

    /**
     * Check if payroll is associated with a ledger
     */
    public function hasLedger()
    {
        return !is_null($this->Lg_Id);
    }

    /**
     * Get ledger name if associated
     */
    public function getLedgerNameAttribute()
    {
        return $this->ledger ? $this->ledger->Lg_Name : null;
    }

    /**
     * Get ledger code if associated
     */
    public function getLedgerCodeAttribute()
    {
        return $this->ledger ? $this->ledger->Lg_Code : null;
    }
}
