<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class SubLedgerMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'subledg_master';
    protected $primaryKey = 'SL_Id';

    protected $fillable = [
        'Lg_ID',
        'SL_Code',
        'SL_Name',
        'SL_Address',
        'SL_City',
        'SL_Pin',
        'SL_Phone',
        'SL_Email',
        'SL_Cont_Pers',
        'SL_Cr_Lim',
        'SL_Draw_Pwr',
        'SL_Status',
        // Statutory & Legal Details
        'PAN_Number',
        'PAN_Verified',
        'PAN_Verified_Date',
        'GSTIN',
        'GSTIN_Verified',
        'GSTIN_Verified_Date',
        'GST_Exempt_Declaration',
        'MSME_Udyam_Registration_No',
        'CIN_LLPIN',
        'Shop_Act_Trade_License_No',
        'TAN',
        // Banking & Payment Details
        'Bank_Name',
        'Branch_Name',
        'Account_Holder_Name',
        'Account_Number',
        'IFSC_Code',
        'Account_Type',
        'Cancelled_Cheque_Path',
        'Cancelled_Cheque_Original_Name',
        'Bank_Details_Verified',
        'Bank_Details_Verified_Date'
    ];

    protected $casts = [
        'SL_Cr_Lim' => 'decimal:2',
        'SL_Draw_Pwr' => 'decimal:2',
        'SL_Status' => 'boolean',
        'PAN_Verified' => 'boolean',
        'GSTIN_Verified' => 'boolean',
        'Bank_Details_Verified' => 'boolean',
        'PAN_Verified_Date' => 'date',
        'GSTIN_Verified_Date' => 'date',
        'Bank_Details_Verified_Date' => 'date'
    ];

    // Validation rules for create
    public static function createRules()
    {
        return [
            'Lg_ID' => 'required|exists:ledger_master,Lg_Id',
            'SL_Code' => 'nullable|string|max:50|unique:subledg_master,SL_Code',
            'SL_Name' => 'required|string|max:200',
            'SL_Address' => 'nullable|string',
            'SL_City' => 'nullable|string|max:100',
            'SL_Pin' => 'nullable|string|max:20',
            'SL_Phone' => 'nullable|string|max:20',
            'SL_Email' => 'nullable|email|max:100',
            'SL_Cont_Pers' => 'nullable|string|max:100',
            'SL_Cr_Lim' => 'nullable|numeric|min:0|max:9999999999999.99',
            'SL_Draw_Pwr' => 'nullable|numeric|min:0|max:9999999999999.99',
            'SL_Status' => 'nullable|boolean',

            // Statutory & Legal Details validation
            'PAN_Number' => 'nullable|string|regex:/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/|unique:subledg_master,PAN_Number',
            'GSTIN' => 'nullable|string|regex:/^[0-9]{2}[A-Z]{5}[0-9]{4}[A-Z]{1}[1-9A-Z]{1}Z[0-9A-Z]{1}$/|unique:subledg_master,GSTIN',
            'GST_Exempt_Declaration' => 'nullable|string|max:255',
            'MSME_Udyam_Registration_No' => 'nullable|string|regex:/^UDYAM-[A-Z]{2}-[0-9]{2}-[0-9]{7}$/|unique:subledg_master,MSME_Udyam_Registration_No',
            'CIN_LLPIN' => 'nullable|string|regex:/^[UL][0-9]{5}[A-Z]{2}[0-9]{4}[A-Z]{3}[0-9]{6}$/|unique:subledg_master,CIN_LLPIN',
            'Shop_Act_Trade_License_No' => 'nullable|string|max:50',
            'TAN' => 'nullable|string|regex:/^[A-Z]{4}[0-9]{5}[A-Z]{1}$/|unique:subledg_master,TAN',

            // Banking & Payment Details validation
            'Bank_Name' => 'nullable|string|max:100',
            'Branch_Name' => 'nullable|string|max:100',
            'Account_Holder_Name' => 'nullable|string|max:150',
            'Account_Number' => 'nullable|string|digits_between:9,18|unique:subledg_master,Account_Number',
            'IFSC_Code' => 'nullable|string|regex:/^[A-Z]{4}0[A-Z0-9]{6}$/',
            'Account_Type' => 'nullable|in:Savings,Current,OD,CC,NRI,Others',

            // File upload validation
            'cancelled_cheque' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120', // 5MB

            // Verification fields
            'PAN_Verified' => 'nullable|boolean',
            'GSTIN_Verified' => 'nullable|boolean',
            'Bank_Details_Verified' => 'nullable|boolean'
        ];
    }

    // Validation rules for update
    public static function updateRules($id)
    {
        return [
            'Lg_ID' => 'sometimes|required|exists:ledger_master,Lg_Id',
            'SL_Code' => 'sometimes|required|string|max:50|unique:subledg_master,SL_Code,' . $id . ',SL_Id',
            'SL_Name' => 'sometimes|required|string|max:200',
            'SL_Address' => 'nullable|string',
            'SL_City' => 'nullable|string|max:100',
            'SL_Pin' => 'nullable|string|max:20',
            'SL_Phone' => 'nullable|string|max:20',
            'SL_Email' => 'nullable|email|max:100',
            'SL_Cont_Pers' => 'nullable|string|max:100',
            'SL_Cr_Lim' => 'nullable|numeric|min:0|max:9999999999999.99',
            'SL_Draw_Pwr' => 'nullable|numeric|min:0|max:9999999999999.99',
            'SL_Status' => 'nullable|boolean',

            // Statutory & Legal Details validation
            'PAN_Number' => 'nullable|string|regex:/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/|unique:subledg_master,PAN_Number,' . $id . ',SL_Id',
            'GSTIN' => 'nullable|string|regex:/^[0-9]{2}[A-Z]{5}[0-9]{4}[A-Z]{1}[1-9A-Z]{1}Z[0-9A-Z]{1}$/|unique:subledg_master,GSTIN,' . $id . ',SL_Id',
            'GST_Exempt_Declaration' => 'nullable|string|max:255',
            'MSME_Udyam_Registration_No' => 'nullable|string|regex:/^UDYAM-[A-Z]{2}-[0-9]{2}-[0-9]{7}$/|unique:subledg_master,MSME_Udyam_Registration_No,' . $id . ',SL_Id',
            'CIN_LLPIN' => 'nullable|string|regex:/^[UL][0-9]{5}[A-Z]{2}[0-9]{4}[A-Z]{3}[0-9]{6}$/|unique:subledg_master,CIN_LLPIN,' . $id . ',SL_Id',
            'Shop_Act_Trade_License_No' => 'nullable|string|max:50',
            'TAN' => 'nullable|string|regex:/^[A-Z]{4}[0-9]{5}[A-Z]{1}$/|unique:subledg_master,TAN,' . $id . ',SL_Id',

            // Banking & Payment Details validation
            'Bank_Name' => 'nullable|string|max:100',
            'Branch_Name' => 'nullable|string|max:100',
            'Account_Holder_Name' => 'nullable|string|max:150',
            'Account_Number' => 'nullable|string|digits_between:9,18|unique:subledg_master,Account_Number,' . $id . ',SL_Id',
            'IFSC_Code' => 'nullable|string|regex:/^[A-Z]{4}0[A-Z0-9]{6}$/',
            'Account_Type' => 'nullable|in:Savings,Current,OD,CC,NRI,Others',

            // File upload validation
            'cancelled_cheque' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120', // 5MB

            // Verification fields
            'PAN_Verified' => 'nullable|boolean',
            'GSTIN_Verified' => 'nullable|boolean',
            'Bank_Details_Verified' => 'nullable|boolean'
        ];
    }

    // Custom validation messages
    public static function validationMessages()
    {
        return [
            'Lg_ID.required' => 'Ledger is required',
            'Lg_ID.exists' => 'Selected ledger does not exist',
            'SL_Code.required' => 'Sub-ledger code is required',
            'SL_Code.unique' => 'This sub-ledger code already exists',
            'SL_Code.max' => 'Sub-ledger code cannot exceed 50 characters',
            'SL_Name.required' => 'Sub-ledger name is required',
            'SL_Name.max' => 'Sub-ledger name cannot exceed 200 characters',
            'SL_Email.email' => 'Please enter a valid email address',
            'SL_Cr_Lim.numeric' => 'Credit limit must be a number',
            'SL_Draw_Pwr.numeric' => 'Drawing power must be a number',

            // Statutory & Legal validation messages
            'PAN_Number.regex' => 'PAN Number must be in valid format (e.g., ABCDE1234F)',
            'PAN_Number.unique' => 'This PAN Number already exists in our records',
            'GSTIN.regex' => 'GSTIN must be in valid format (22 characters starting with state code)',
            'GSTIN.unique' => 'This GSTIN already exists in our records',
            'MSME_Udyam_Registration_No.regex' => 'MSME Udyam Registration must be in format: UDYAM-XX-00-0000000',
            'MSME_Udyam_Registration_No.unique' => 'This MSME Udyam Registration already exists',
            'CIN_LLPIN.regex' => 'CIN/LLPIN must be in valid format (21 characters)',
            'CIN_LLPIN.unique' => 'This CIN/LLPIN already exists',
            'TAN.regex' => 'TAN must be in valid format (e.g., ABCD12345E)',
            'TAN.unique' => 'This TAN already exists',

            // Banking & Payment validation messages
            'Account_Number.digits_between' => 'Account number must be between 9 to 18 digits',
            'Account_Number.unique' => 'This account number already exists',
            'IFSC_Code.regex' => 'IFSC Code must be in valid format (11 characters, e.g., SBIN0001234)',
            'Account_Type.in' => 'Account type must be one of: Savings, Current, OD, CC, NRI, Others',

            // File validation messages
            'cancelled_cheque.mimes' => 'Cancelled cheque must be PDF, JPG, JPEG or PNG file',
            'cancelled_cheque.max' => 'Cancelled cheque file size must be less than 5MB'
        ];
    }

    // Relationships
    public function ledger()
    {
        return $this->belongsTo(LedgerMaster::class, 'Lg_ID', 'Lg_Id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('SL_Status', true);
    }

    public function scopeByLedger($query, $ledgerId)
    {
        return $query->where('Lg_ID', $ledgerId);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('SL_Code', 'like', "%{$search}%")
                ->orWhere('SL_Name', 'like', "%{$search}%")
                ->orWhere('SL_Cont_Pers', 'like', "%{$search}%")
                ->orWhere('SL_City', 'like', "%{$search}%")
                ->orWhere('SL_Phone', 'like', "%{$search}%")
                ->orWhere('SL_Email', 'like', "%{$search}%")
                ->orWhere('PAN_Number', 'like', "%{$search}%")
                ->orWhere('GSTIN', 'like', "%{$search}%")
                ->orWhere('Account_Number', 'like', "%{$search}%")
                ->orWhereHas('ledger', function ($q) use ($search) {
                    $q->where('Lg_Code', 'like', "%{$search}%")
                        ->orWhere('Lg_Name', 'like', "%{$search}%");
                });
        });
    }

    public function scopeWithCreditLimit($query)
    {
        return $query->where('SL_Cr_Lim', '>', 0);
    }

    public function scopeWithDrawingPower($query)
    {
        return $query->where('SL_Draw_Pwr', '>', 0);
    }

    // New scopes for statutory details
    public function scopeHasPAN($query)
    {
        return $query->whereNotNull('PAN_Number')->where('PAN_Verified', true);
    }

    public function scopeHasGSTIN($query)
    {
        return $query->whereNotNull('GSTIN')->where('GSTIN_Verified', true);
    }

    public function scopeHasBankDetails($query)
    {
        return $query->whereNotNull('Account_Number')->where('Bank_Details_Verified', true);
    }

    public function scopeIsMSME($query)
    {
        return $query->whereNotNull('MSME_Udyam_Registration_No');
    }

    // Attribute accessors
    public function getFullAddressAttribute()
    {
        $addressParts = [];
        if ($this->SL_Address) $addressParts[] = $this->SL_Address;
        if ($this->SL_City) $addressParts[] = $this->SL_City;
        if ($this->SL_Pin) $addressParts[] = $this->SL_Pin;

        return implode(', ', $addressParts);
    }

    public function getFormattedCreditLimitAttribute()
    {
        return number_format($this->SL_Cr_Lim, 2);
    }

    public function getFormattedDrawingPowerAttribute()
    {
        return number_format($this->SL_Draw_Pwr, 2);
    }

    public function getStatusTextAttribute()
    {
        return $this->SL_Status ? 'Active' : 'Inactive';
    }

    public function getLedgerNameAttribute()
    {
        return $this->ledger ? $this->ledger->Lg_Name : null;
    }

    public function getLedgerCodeAttribute()
    {
        return $this->ledger ? $this->ledger->Lg_Code : null;
    }

    // New accessors for statutory details
    public function getMaskedPANAttribute()
    {
        if (!$this->PAN_Number) return null;
        return substr($this->PAN_Number, 0, 2) . 'XXXXX' . substr($this->PAN_Number, -2);
    }

    public function getMaskedAccountNumberAttribute()
    {
        if (!$this->Account_Number) return null;
        return 'XXXXXX' . substr($this->Account_Number, -4);
    }

    public function getGSTINStateCodeAttribute()
    {
        if (!$this->GSTIN) return null;
        return substr($this->GSTIN, 0, 2);
    }

    public function getFormattedPANVerifiedDateAttribute()
    {
        return $this->PAN_Verified_Date ? $this->PAN_Verified_Date->format('d/m/Y') : null;
    }

    public function getFormattedGSTINVerifiedDateAttribute()
    {
        return $this->GSTIN_Verified_Date ? $this->GSTIN_Verified_Date->format('d/m/Y') : null;
    }

    public function getFormattedBankDetailsVerifiedDateAttribute()
    {
        return $this->Bank_Details_Verified_Date ? $this->Bank_Details_Verified_Date->format('d/m/Y') : null;
    }

    public function getCancelledChequeUrlAttribute()
    {
        if (!$this->Cancelled_Cheque_Path) return null;
        return asset('storage/' . $this->Cancelled_Cheque_Path);
    }

    // Business logic methods
    public function deactivate()
    {
        $this->update(['SL_Status' => false]);
        return $this;
    }

    public function activate()
    {
        $this->update(['SL_Status' => true]);
        return $this;
    }

    public function getAvailableCredit()
    {
        return max(0, $this->SL_Cr_Lim - $this->SL_Draw_Pwr);
    }

    // New business logic methods for statutory details
    public function verifyPAN()
    {
        $this->update([
            'PAN_Verified' => true,
            'PAN_Verified_Date' => now()
        ]);
        return $this;
    }

    public function verifyGSTIN()
    {
        $this->update([
            'GSTIN_Verified' => true,
            'GSTIN_Verified_Date' => now()
        ]);
        return $this;
    }

    public function verifyBankDetails()
    {
        $this->update([
            'Bank_Details_Verified' => true,
            'Bank_Details_Verified_Date' => now()
        ]);
        return $this;
    }

    public function unverifyPAN()
    {
        $this->update([
            'PAN_Verified' => false,
            'PAN_Verified_Date' => null
        ]);
        return $this;
    }

    public function unverifyGSTIN()
    {
        $this->update([
            'GSTIN_Verified' => false,
            'GSTIN_Verified_Date' => null
        ]);
        return $this;
    }

    public function unverifyBankDetails()
    {
        $this->update([
            'Bank_Details_Verified' => false,
            'Bank_Details_Verified_Date' => null
        ]);
        return $this;
    }

    public function hasCompleteStatutoryDetails()
    {
        return !empty($this->PAN_Number) && !empty($this->GSTIN) && $this->PAN_Verified && $this->GSTIN_Verified;
    }

    public function hasCompleteBankingDetails()
    {
        return !empty($this->Account_Number) &&
            !empty($this->Account_Holder_Name) &&
            !empty($this->IFSC_Code) &&
            !empty($this->Bank_Name) &&
            $this->Bank_Details_Verified;
    }

    public function getComplianceStatusAttribute()
    {
        $score = 0;
        $total = 0;

        // PAN verification
        if ($this->PAN_Number) {
            $total++;
            if ($this->PAN_Verified) $score++;
        }

        // GSTIN verification
        if ($this->GSTIN) {
            $total++;
            if ($this->GSTIN_Verified) $score++;
        }

        // Bank details verification
        if ($this->Account_Number) {
            $total++;
            if ($this->Bank_Details_Verified) $score++;
        }

        return $total > 0 ? round(($score / $total) * 100, 2) : 0;
    }

    public function getComplianceStatusTextAttribute()
    {
        $score = $this->compliance_status;

        if ($score >= 90) return 'Excellent';
        if ($score >= 75) return 'Good';
        if ($score >= 50) return 'Average';
        if ($score >= 25) return 'Poor';
        return 'Very Poor';
    }

    public function getBankDetailsSummaryAttribute()
    {
        if (!$this->Account_Number) return null;

        return [
            'bank_name' => $this->Bank_Name,
            'branch_name' => $this->Branch_Name,
            'account_holder_name' => $this->Account_Holder_Name,
            'masked_account_number' => $this->masked_account_number,
            'ifsc_code' => $this->IFSC_Code,
            'account_type' => $this->Account_Type,
            'verified' => $this->Bank_Details_Verified,
            'verified_date' => $this->formatted_bank_details_verified_date
        ];
    }

    public function getStatutoryDetailsSummaryAttribute()
    {
        $details = [];

        if ($this->PAN_Number) {
            $details['pan'] = [
                'number' => $this->masked_pan,
                'verified' => $this->PAN_Verified,
                'verified_date' => $this->formatted_pan_verified_date
            ];
        }

        if ($this->GSTIN) {
            $details['gstin'] = [
                'number' => $this->GSTIN,
                'state_code' => $this->gstin_state_code,
                'verified' => $this->GSTIN_Verified,
                'verified_date' => $this->formatted_gstin_verified_date
            ];
        }

        if ($this->MSME_Udyam_Registration_No) {
            $details['msme'] = [
                'registration_number' => $this->MSME_Udyam_Registration_No
            ];
        }

        if ($this->CIN_LLPIN) {
            $details['company'] = [
                'cin_llpin' => $this->CIN_LLPIN
            ];
        }

        if ($this->TAN) {
            $details['tan'] = [
                'number' => $this->TAN
            ];
        }

        return $details;
    }
}
