<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class UserRoleMapping extends Model
{
    use HasFactory;

    protected $table = 'user_role_mapping';
    protected $primaryKey = 'URoleMap';
    public $timestamps = true;

    protected $fillable = [
        'U_Id',
        'Role_Id',
        'Default_Rid'
    ];

    protected $casts = [
        'Default_Rid' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relationship with User
    public function user()
    {
        return $this->belongsTo(GmcUser::class, 'U_Id', 'user_id');
    }

    // Relationship with Role
    public function role()
    {
        return $this->belongsTo(RoleMaster::class, 'Role_Id', 'Role_Id');
    }

    // Scope for default role
    public function scopeDefaultRole($query)
    {
        return $query->where('Default_Rid', 1);
    }

    // Scope for user roles
    public function scopeForUser($query, $userId)
    {
        return $query->where('U_Id', $userId);
    }

    // Check if role is default
    public function isDefault()
    {
        return $this->Default_Rid === 1;
    }

    // Set as default role (and unset others)
    public function setAsDefault()
    {
        // Remove default from other roles of same user
        self::where('U_Id', $this->U_Id)
            ->where('URoleMap', '!=', $this->URoleMap)
            ->update(['Default_Rid' => 0]);

        // Set this as default
        $this->update(['Default_Rid' => 1]);
    }

    // Get default role for user
    public static function getDefaultRoleForUser($userId)
    {
        return self::with('role')
            ->where('U_Id', $userId)
            ->where('Default_Rid', 1)
            ->first();
    }

    // Get all roles for user
    public static function getAllRolesForUser($userId)
    {
        return self::with('role')
            ->where('U_Id', $userId)
            ->orderBy('Default_Rid', 'desc')
            ->get();
    }

    // Check if user has role
    public static function userHasRole($userId, $roleId)
    {
        return self::where('U_Id', $userId)
            ->where('Role_Id', $roleId)
            ->exists();
    }
}
