<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class WardMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'ward_master';
    protected $primaryKey = 'Ward_Id';

    protected $fillable = [
        'Ward_Code',
        'Ward_Name',
        'Zone_Id',
        'is_active'
    ];

    protected $casts = [
        'Ward_Id' => 'integer',
        'Zone_Id' => 'integer',
        'is_active' => 'boolean'
    ];

    // Validation rules for create
    public static function createRules()
    {
        return [
            'Ward_Code' => 'required|string|max:5',
            'Ward_Name' => 'required|string|max:25',
            'Zone_Id' => 'required|exists:zone_masters,Zone_Id'
        ];
    }

    // Validation rules for update
    public static function updateRules($id)
    {
        return [
            'Ward_Code' => 'sometimes|required|string|max:5',
            'Ward_Name' => 'sometimes|required|string|max:25',
            'Zone_Id' => 'sometimes|required|exists:zone_masters,Zone_Id',
            'is_active' => 'sometimes|boolean'
        ];
    }

    // Custom validation messages
    public static function validationMessages()
    {
        return [
            'Ward_Code.required' => 'Ward code is required',
            'Ward_Code.max' => 'Ward code cannot exceed 5 characters',
            'Ward_Code.unique' => 'This ward code already exists in the selected zone',
            'Ward_Name.required' => 'Ward name is required',
            'Ward_Name.max' => 'Ward name cannot exceed 25 characters',
            'Zone_Id.required' => 'Zone is required',
            'Zone_Id.exists' => 'The selected zone does not exist'
        ];
    }

    // Custom validation for unique Ward_Code within zone
    public static function validateUniqueWardCode($wardCode, $zoneId, $excludeId = null)
    {
        $query = self::where('Ward_Code', $wardCode)
                    ->where('Zone_Id', $zoneId);

        if ($excludeId) {
            $query->where('Ward_Id', '!=', $excludeId);
        }

        return !$query->exists();
    }

    // Relationships
    public function zone()
    {
        return $this->belongsTo(ZoneMaster::class, 'Zone_Id', 'Zone_Id');
    }

    public function cccs()
    {
        return $this->hasMany(CccMaster::class, 'Ward_Id', 'Ward_Id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByZone($query, $zoneId)
    {
        return $query->where('Zone_Id', $zoneId);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('Ward_Code', 'like', "%{$search}%")
              ->orWhere('Ward_Name', 'like', "%{$search}%")
              ->orWhereHas('zone', function ($q) use ($search) {
                  $q->where('Zone_Code', 'like', "%{$search}%")
                    ->orWhere('Zone_Name', 'like', "%{$search}%");
              });
        });
    }

    // Attribute accessors
    public function getFullWardAttribute()
    {
        return $this->Ward_Code . ' - ' . $this->Ward_Name;
    }

    public function getZoneNameAttribute()
    {
        return $this->zone ? $this->zone->Zone_Name : null;
    }

    public function getZoneCodeAttribute()
    {
        return $this->zone ? $this->zone->Zone_Code : null;
    }

    // Business logic methods
    public function canBeDeleted()
    {
        // Check if ward has dependencies (like CCCs)
        return $this->cccs()->count() === 0;
    }

    public function deactivate()
    {
        $this->update(['is_active' => false]);
        return $this;
    }

    public function activate()
    {
        $this->update(['is_active' => true]);
        return $this;
    }
}
